# -*- coding: utf-8 -*-

from collections import namedtuple

from lib import xdb
from lib.cache import memoize
from lib.xdb import queries

MailMessage = namedtuple(
    "MailMessage",
    ["mid", "st_id", "size", "attributes", "attaches", "subject",
     "firstline", "hdr_date", "hdr_message_id", "recipients",
     "extra_data", "pop_uidl", "found_tid", "thread_rule", "mime"])

MailBox = namedtuple(
    "MailBox",
    ["uid", "mid", "fid", "tid", "newest_tif", "imap_id", "revision",
     "chain", "seen", "recent", "deleted", "received_date",
     "lids", "doom_date"])

MailishMessage = namedtuple(
    "MailishMessage",
    ["uid", "fid", "imap_id", "imap_time", "mid", "errors"])

DeletedBox = namedtuple(
    "DeletedBox",
    ["deleted_date", "uid", "mid", "revision", "info"])

MailLabels = namedtuple(
    "MailLabels",
    ["lid", "name", "type", "color", "created", "message_count"])

MailFolder = namedtuple(
    "MailFolder",
    ["fid", "revision", "name", "parent_fid", "type", "unvisited",
     "unique_type", "created", "next_imap_id",
     "uidvalidity", "first_unseen", "first_unseen_id",
     "message_count", "message_seen", "message_size",
     "message_recent", "attach_count", "attach_size",
     "pop3state", "position", "subscribed_for_shared_folder"])

ChangeLogEntry = namedtuple(
    "ChangeLogEntry",
    ["cid", "uid", "revision", "type", "change_date", "changed", "arguments",
     "fresh_count", "useful_new_count", "ds_user", "x_request_id", "session_key", "quiet"]
)

_cache = {}


@memoize(_cache)
def find_fid_by_path(uid, path):
    return find_folder_by_path(uid, path).fid


def find_mail_message_row_by_subject(uid, subject):
    return find_mail_message_row(uid, subject=subject)


def find_mail_message_row_by_hdr_message_id(uid, hdr_message_id):
    return find_mail_message_row(uid, hdr_message_id=hdr_message_id)


def find_mail_message_row_by_mid(uid, mid):
    return find_mail_message_row(uid, mid=mid)


def find_mail_message_row(uid, **kwargs):
    if len(kwargs) != 1:
        raise ValueError("Specify where condition, e.g. field=value")

    field, value = kwargs.items()[0]

    sql = queries.messages_by_field(uid, field, value)
    row = xdb.fetch_one(uid, sql)
    if row:
        return MailMessage(*row)
    return None


def find_mail_box_row_by_mid(uid, mid):
    sql = queries.mailbox(uid, mid)
    row = xdb.fetch_one(uid, sql)
    if row:
        return MailBox(*row)
    return None


def find_mail_box_row_by_message_id(uid, msg_id):
    mail_message_row = find_mail_message_row(uid, hdr_message_id=msg_id)
    if mail_message_row is None:
        raise RuntimeError("Message for uid={0} and msg_id={1} not found in mail.messages".format(uid, msg_id))
    return find_mail_box_row_by_mid(uid, mail_message_row.mid)


def find_deleted_box_row_by_mid(uid, mid):
    sql = queries.deleted_box(uid, mid)
    row = xdb.fetch_one(uid, sql)
    if row:
        return DeletedBox(*row)
    return None


def find_mail_message_rows_by_field(uid, field, value):
    sql = queries.messages_by_field(uid, field, value)
    rows = xdb.fetch_all(uid, sql)
    return [MailMessage(*row) for row in rows]


def find_mail_message_rows(uid, **kwargs):
    if len(kwargs) != 1:
        raise ValueError("Specify where condition, e.g. field=value")

    field, value = kwargs.items()[0]
    return find_mail_message_rows_by_field(uid, field, value)


def find_mail_label_rows_by_field(uid, field_name, field_value):
    sql = queries.mail_labels_by_field(uid, field_name, field_value)
    rows = xdb.fetch_all(uid, sql)
    return [MailLabels(*row) for row in rows]


def find_mail_label_rows_by_name(uid, name):
    return find_mail_label_rows_by_field(uid, "name", name)


def find_mail_label_rows_by_fields(uid, field_name, field_values):
    sql = queries.mail_labels_by_fields(uid, field_name, field_values)
    rows = xdb.fetch_all(uid, sql)
    return [MailLabels(*row) for row in rows]


def find_several_mail_label_rows(uid, count):
    sql = queries.several_mail_labels(uid, count)
    rows = xdb.fetch_all(uid, sql)
    return [MailLabels(*row) for row in rows]


def find_box_rows_by_subject(uid, subject, deleted=False):
    find_box = find_mail_box_row_by_mid
    if deleted:
        find_box = find_deleted_box_row_by_mid

    rows = find_mail_message_rows_by_field(uid, "subject", subject)
    return [find_box(uid, row.mid) for row in rows]


def find_folder_by_path(uid, path):
    parent_fid = None
    folder = None

    for name in path.split("|"):
        sql = queries.mail_folders_by_fields(uid=uid, parent_fid=parent_fid, name=name)
        row = xdb.fetch_one(uid, sql)
        if not row:
            return None
        folder = MailFolder(*row)
        parent_fid = folder.fid

    return folder


def find_folder_by_fid(uid, fid):
    sql = queries.mail_folders_by_fields(uid=uid, fid=fid)
    row = xdb.fetch_one(uid, sql)
    if not row:
        return None
    return MailFolder(*row)


def find_mail_changelog_entry_by_mid(uid, mid):
    sql = queries.mail_changelog_entry_by_mid(uid=uid, mid=mid)
    row = xdb.fetch_one(uid, sql)
    if not row:
        return None
    return ChangeLogEntry(*row)


def find_mailish_message_by_mid(uid, mid):
    sql = queries.mailish_message_by_mid(uid=uid, mid=mid)
    row = xdb.fetch_one(uid, sql)
    if not row:
        return None
    return MailishMessage(*row)


def get_fids_of_messages_with_msg_id(user, msg_id):
    rows = find_mail_message_rows(user.uid, hdr_message_id=msg_id)
    fids = [find_mail_box_row_by_mid(user.uid, row.mid).fid for row in rows]
    return sorted(fids)
