# -*- coding: utf-8 -*-

import os
import yaml
from dns import resolver, reversename

SETTINGS = {}
_MX_TYPE = None
_ENV_TYPE = None
_MAIL_TYPE = None


def get_mx_settings():
    return SETTINGS[_MX_TYPE]


def get_mx_type():
    return _MX_TYPE


def get_mail_type():
    return _MAIL_TYPE


def get_env_type():
    return _ENV_TYPE


def get_host():
    return SETTINGS[_MX_TYPE]["host"]


def discover_dns_host(host):
    """
    Return !FIRST! fqdn from (Qloud) DNS dicsovery record:
    Example in bash:
    $ host yaback.qa.nwsmtp.mail.stable.qloud-d.yandex.net
    yaback.qa.nwsmtp.mail.stable.qloud-d.yandex.net has IPv6 address 2a02:6b8:c14:5d22::640:cd2e:e94a
    $ host 2a02:6b8:c14:5d22::640:cd2e:e94a
    a.4.9.e.e.2.d.c.0.4.6.0.0.0.0.0.2.2.d.5.4.1.c.0.8.b.6.0.2.0.a.2.ip6.arpa domain name pointer sas1-cd2ee94a429d.qloud-c.yandex.net.
    """
    dns_resolver = resolver.Resolver()
    ipv6 = dns_resolver.query(host, "AAAA")[0]
    ipv6_reverse = reversename.from_address(ipv6.to_text())
    host_fqdn = dns_resolver.query(ipv6_reverse, "PTR")[0].to_text()
    return host_fqdn[:-1]


def initialize(base_dir, mx, env, mail, mx_host, mx_port):
    global _MX_TYPE
    global _ENV_TYPE
    global _MAIL_TYPE
    _MX_TYPE = mx
    _ENV_TYPE = env
    _MAIL_TYPE = mail

    global SETTINGS
    SETTINGS = read_settings(base_dir, env, mail)

    if mx_host is None:
        mx_host = SETTINGS[_MX_TYPE]["host"]
    mx_host = discover_dns_host(mx_host)
    update_mx_host_port(mx_host, mx_port)


def read_settings(base_dir, env, mail):
    base_path = os.path.join(base_dir, "settings.yaml")
    override_path = os.path.join(
        base_dir,
        "settings.{env}.{mail}.yaml".format(env=env, mail=mail)
    )
    base = yaml.safe_load(open(base_path))
    _merge(base, yaml.safe_load(open(override_path)))

    return base


def update_mx_host_port(mx_host, mx_port):
    if mx_host:
        SETTINGS[_MX_TYPE]["host"] = mx_host
    if mx_port:
        SETTINGS[_MX_TYPE]["port"] = mx_port


def _merge(left, right):
    """
    >>> _merge({}, {})
    {}
    >>> _merge({1: 2}, {})
    {1: 2}
    >>> _merge({1: 2}, {1: 3})
    {1: 3}
    >>> _merge({1: {2: 3}}, {1: {2: 4}})
    {1: {2: 4}}
    >>> _merge({1: {}}, {1: {2: 4}})
    {1: {2: 4}}
    >>> _merge({1: {}, 3: 5}, {1: {2: 4}})
    {1: {2: 4}, 3: 5}
    >>> _merge({1: 10}, {1: {2: 4}})
    Traceback (most recent call last):
        ...
    AttributeError: 'int' object has no attribute 'setdefault'
    """
    for k, v in right.iteritems():
        left.setdefault(k, {})
        if isinstance(v, dict):
            _merge(left[k], v)
        else:
            left[k] = v
