# -*- coding: utf-8 -*-

import logging
import smtplib
from StringIO import StringIO
from contextlib import contextmanager
from email.message import Message
from email.utils import make_msgid
from functools import partial, wraps

import pytest
from hamcrest import assert_that, equal_to

from lib.headers import Header
from lib.msgs_builder import build_plain_text_message, IMPORTANT_TEXT
from lib.settings import SETTINGS, get_mx_settings
from lib.utils import allure_attach

MX_SETTINGS = get_mx_settings()
HOST = MX_SETTINGS["host"]
PORT = MX_SETTINGS["port"]
USE_TLS = MX_SETTINGS["use_tls"]
USE_AUTH = MX_SETTINGS["use_auth"]
SMTPLIB_LOG_MAX_LINE_SIZE = SETTINGS["logging"]["smtplib"]["max_size"]


class LogRedirectingStringIO(StringIO):
    def __init__(self, logger_name, max_size, *args, **kwargs):
        StringIO.__init__(self, *args, **kwargs)
        self.__log = logging.getLogger(logger_name)
        self.__pos = 0
        assert max_size > 0
        self.__max_size = max_size

    def write(self, s):
        StringIO.write(self, s)
        if s != '\n':
            return
        self.seek(self.__pos)
        for line in self.readlines():
            msg = line.strip()
            if len(msg) > self.__max_size:
                msg = msg[:self.__max_size] + "..."
            self.__log.info(msg)
        self.__pos = self.tell()
        assert len(self.buflist) == 0


def patch_smtplib_logging():
    smtplib.stderr = LogRedirectingStringIO("smtplib", SMTPLIB_LOG_MAX_LINE_SIZE)


def default_allure_factory(*args, **kwargs):
    return partial(allure_attach, name="SMTP SESSION")


class CaptureHandler(logging.Handler):
    def __init__(self, *args, **kwargs):
        super(CaptureHandler, self).__init__(*args, **kwargs)
        self.buffer = StringIO()

    def emit(self, record):
        self.buffer.write(record.getMessage())


@contextmanager
def capture_smtp_logging():
    handler = CaptureHandler()
    smtplog = logging.getLogger("smtplib")
    smtplog.addHandler(handler)
    try:
        yield handler.buffer
    finally:
        smtplog.removeHandler(handler)


@contextmanager
def _assert_smtp_session_is_ok():
    try:
        yield
    except smtplib.SMTPException as e:
        pytest.fail(str(e))


def assert_smtp_session_is_ok(func):
    @wraps(func)
    def decorated(*args, **kwargs):
        with _assert_smtp_session_is_ok():
            return func(*args, **kwargs)

    return decorated


@contextmanager
def _allure_attach_smtp_logging(allure_factory=default_allure_factory):
    with capture_smtp_logging() as buf:
        try:
            yield
        finally:
            allure_attach = allure_factory()
            allure_attach(buf.getvalue())


def allure_attach_smtp_logging(func):
    @wraps(func)
    def decorated(*args, **kwargs):
        with _allure_attach_smtp_logging():
            return func(*args, **kwargs)

    return decorated


@contextmanager
def create_smtp_session(host=HOST, port=PORT, use_ssl=False, use_tls=USE_TLS, use_auth=USE_AUTH, auth=None, debug=True):
    if use_ssl and use_tls:
        raise RuntimeError("SSL and TLS cannot be used together")
    if use_ssl:
        client = smtplib.SMTP_SSL(host, port)
    else:
        client = smtplib.SMTP(host, port)
    client.set_debuglevel(debug)
    if use_tls:
        client.starttls()
    if use_auth:
        client.login(**auth)

    try:
        yield client
    finally:
        client.quit()


@allure_attach_smtp_logging
def auth_login_on_smtp_server(user, password):
    with create_smtp_session(auth={"user": user, "password": password}, use_auth=True) as session:
        return session


def send_letter_to_outside(sender, rcpts, msg, check=True):
    smtp_params = SETTINGS["external_smtp"][sender.domain.replace('.', '_')]
    host = smtp_params["host"]
    port = smtp_params["port"]
    use_ssl = smtp_params["use_ssl"]
    use_tls = smtp_params["use_tls"]
    use_auth = True

    return send_letter(sender, rcpts, msg, host, port, use_ssl, use_tls, use_auth, check)


@assert_smtp_session_is_ok
@allure_attach_smtp_logging
def send_letter(sender, rcpts, msg, host=HOST, port=PORT, use_ssl=False,
                use_tls=USE_TLS, use_auth=USE_AUTH, check=True):
    if sender is None:
        user = ""
        password = ""
        mail_from = ""
    else:
        user = sender.email
        password = sender.pwd
        mail_from = sender.email

    if isinstance(msg, Message):
        msg = msg.as_string()

    with create_smtp_session(host, port, use_ssl, use_tls, use_auth,
                             auth={"user": user, "password": password}) as session:
        session.ehlo("qwerty")
        session.mail(mail_from)
        if not isinstance(rcpts, list):
            rcpts = [rcpts]
        for rcpt in rcpts:
            session.rcpt(rcpt)
        errcode, errmsg = session.data(msg)
        if check:
            assert_that(errcode, equal_to(250))
        return errcode, errmsg


def send_plain_text_message(sender, to_emails, headers=None, subject=None, msg_id=None, text="some text", check=True):
    if headers is None:
        headers = {}
    if msg_id is None:
        msg_id = make_msgid()
    headers[Header.MESSAGE_ID] = msg_id
    if subject is not None:
        headers[Header.SUBJECT] = subject
    msg = build_plain_text_message(from_email=sender.email, to_emails=to_emails, text=text, headers=headers)
    send_letter(sender, to_emails, msg.as_string(), check=check)
    return msg_id


def send_important_message(sender, to_emails, headers=None):
    # https://st.yandex-team.ru/SODEV-2149
    return send_plain_text_message(sender, to_emails, text=IMPORTANT_TEXT, headers=headers)


@allure_attach_smtp_logging
def provide_smtp_session_until_rcpt_to_command(sender, rcpt_email):
    with create_smtp_session(auth={"user": sender.email, "password": sender.pwd}) as session:
        session.ehlo("qwerty")
        session.mail(sender.email)
        smtp_code, smtp_msg = session.rcpt(rcpt_email)
        return smtp_code, smtp_msg
