# -*- coding: utf-8 -*-

import StringIO
import logging
import re
import time

import paramiko
from library.python.vault_client import instances

from lib.settings import SETTINGS

PROXY_HOST = SETTINGS["ssh_proxy"]["host"]
PRIVATE_KEY_VERSION = SETTINGS["private_key"]["version"]

logging.getLogger("paramiko").addHandler(logging.NullHandler())
logging.getLogger("paramiko").propagate = 0
logger = logging.getLogger("ssh")

_cache = {}


def get_ssh_key_from_secret_store():
    client = instances.Production()
    retrieved_version = client.get_version(PRIVATE_KEY_VERSION)
    return retrieved_version["value"]["private_key"].strip()


def get_sshclient(host=PROXY_HOST):
    """Returns ssh client connected to specified host."""
    global _cache
    if host in _cache:
        return _cache[host]

    sshclient = paramiko.SSHClient()
    sshclient.set_missing_host_key_policy(paramiko.AutoAddPolicy())
    private_key = paramiko.RSAKey.from_private_key(StringIO.StringIO(get_ssh_key_from_secret_store()))
    sshclient.connect(host, username="root", pkey=private_key)

    _cache[host] = sshclient
    return sshclient


def _shadow_pwd(s):
    return re.sub(r"password=[^\s]+", "password=***", s)


def exec_command(sshclient, cmd):
    """Wrapper for `paramiko.SSHClient.exec_command`, which is doing blocking call."""
    logger.info(_shadow_pwd(cmd))
    try:
        _, stdout, stderr = sshclient.exec_command(cmd)
        _wait_for_command(cmd, stderr)
        return stdout, stderr
    except Exception as e:
        raise Exception(_shadow_pwd(e.message))


def _recv_all_stderr(ssh_channel):
    buffer_len = 1024
    stderr = ""
    buff = ssh_channel.recv_stderr(buffer_len)
    while buff:
        stderr += buff
        buff = ssh_channel.recv_stderr(buffer_len)
    return stderr


def _wait_for_command(cmd, stderr):
    """Wait for command to complete."""
    while not stderr.channel.exit_status_ready():
        time.sleep(0.1)

    if stderr.channel.exit_status:
        stderr_output = _recv_all_stderr(stderr.channel)
        raise RuntimeError("Got an error on command: {cmd}\nstderr:\n{stderr}"
                           .format(cmd=cmd, stderr=stderr_output))
