package ru.yandex.mail.autouser.cron;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Calendar;
import java.util.stream.Collectors;

import javax.inject.Inject;
import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;

import ru.yandex.mail.autouser.mail.SmtpService;
import ru.yandex.mail.autouser.service.AgentList;
import ru.yandex.mail.autouser.utils.MessageTypes;

/**
 * @author Sergey Galyamichev
 */
@Component
public class HeartbeatJob implements Job {
    private static final Logger LOG = LoggerFactory.getLogger(HeartbeatJob.class);
    private static final String MAIL_SUFFIX = "@yandex.ru";

    @Value("${yandex.username}")
    private String from;

    @Inject
    private AgentList agentList;
    @Inject
    private SmtpService smtpService;

    public void execute(JobExecutionContext context) {
        smtpService.send(this::prepare);
    }

    private void prepare(MimeMessage mimeMessage) throws MessagingException {
        Calendar now = Calendar.getInstance();
        int number = now.get(Calendar.HOUR_OF_DAY) * 60 + now.get(Calendar.MINUTE);
        MimeMessageHelper message = new MimeMessageHelper(mimeMessage, true, StandardCharsets.UTF_8.name());
        message.setFrom(from + MAIL_SUFFIX);
        InternetAddress[] agents = agentList.getAgents();
        message.setTo(agents);
        message.setSubject(MessageTypes.HEARTBEAT.getPrettyName() + " #" + number);
        message.setText("Heartbeat-Start: " + now.getTimeInMillis(), false);
        String agentAddresses = Arrays.stream(agents)
                .map(InternetAddress::toString)
                .collect(Collectors.joining(", "));
        LOG.info("Send heartbeat to all agents [{}] number {}", agentAddresses, number);
    }
}
