package ru.yandex.mail.autouser.cron;

import java.io.IOException;
import java.util.Arrays;

import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;
import javax.ws.rs.core.MediaType;

import com.codahale.metrics.Histogram;
import com.codahale.metrics.MetricRegistry;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.stereotype.Component;

import ru.yandex.mail.autouser.reports.HeartbeatReport;
import ru.yandex.mail.autouser.utils.MessageHelper;

/**
 * @author Sergey Galyamichev
 */
@Component
public class HeartbeatMessageProcessor {
    private static final ObjectMapper OM = new ObjectMapper();

    private final MetricRegistry metricRegistry;

    public HeartbeatMessageProcessor(MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
    }

    public void doProcess(Message message) throws MessagingException {
        try {
            InternetAddress agent = getFrom(message);
            Histogram inDeliveryTime = metricRegistry.histogram(inDeliveryFor(agent.getPersonal())+ ".histo");
            metricRegistry.counter(inDeliveryFor(agent.getPersonal()) + ".counter").inc();
            inDeliveryTime.update(MessageHelper.getDeliveryTime(message));
            String content = getContent(message);
            if (content != null) {
                HeartbeatReport response = OM.readValue(content, HeartbeatReport.class);
                Histogram outDeliveryTime = metricRegistry.histogram(outDeliveryFor(agent.getPersonal()));
                outDeliveryTime.update(response.getDeliveryTime());
            }
        } catch (IOException e) {
            throw new MessagingException("", e);
        }
    }

    private String getContent(Message message) throws MessagingException, IOException {
        if (MediaType.TEXT_PLAIN.equals(message.getContentType())) {
            return message.getContent().toString();
        }
        return null;
    }

    private String inDeliveryFor(String agent) {
        return MetricRegistry.name(ReportingJob.class, agent, "in.delivery.time");
    }
    private String outDeliveryFor(String agent) {
        return MetricRegistry.name(ReportingJob.class, agent, "out.delivery.time");
    }

    private InternetAddress getFrom(Message message) throws MessagingException {
        return (InternetAddress) Arrays.stream(message.getFrom())
                .findFirst()
                .orElseThrow(() -> new MessagingException("Can't find sender"));
    }
}
