package ru.yandex.mail.autouser.service;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;

import com.codahale.metrics.MetricRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

/**
 * @author Sergey Galyamichev
 */
@Component
public class AgentList {
    private static final Logger LOG = LoggerFactory.getLogger(AgentList.class);

    public static final long TTL = 10000;
    private final Map<InternetAddress, Long> agent;
    private MetricRegistry metricRegistry;

    public AgentList(MetricRegistry metricRegistry, @Value("${agents}") String agents) throws AddressException {
        this.metricRegistry = metricRegistry;
        Map<InternetAddress, Long> agent = new HashMap<>();
        Arrays.asList(InternetAddress.parse(agents))
                .forEach(a -> agent.put(a, System.currentTimeMillis()));
        LOG.info("Initial agents are {}", agents);
        this.agent = agent;
    }

    public synchronized String gotPing(String nickname) {
        try {
            InternetAddress agentAddress = new InternetAddress(nickname);
            String name = MetricRegistry.name(AgentList.class, nickname, "ping");
            metricRegistry.counter(name).inc();
            Long oldTtl = agent.put(agentAddress, System.currentTimeMillis() + TTL);
            if (oldTtl != null) {
                LOG.info("TTL for {} updated", nickname);
                return "exists";
            } else {
                LOG.info("TTL for {} added", nickname);
                return "new";
            }
        } catch (AddressException e) {
            return "fails";
        }
    }

    public synchronized InternetAddress[] getAgents() {
        return agent.entrySet().stream()
                .filter(e -> e.getValue() < System.currentTimeMillis())
                .map(Map.Entry::getKey).toArray(InternetAddress[]::new);
    }
}
