package ru.yandex.mail.autouser.commands;

import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;

import ru.yandex.mail.autouser.mail.ImapService;
import ru.yandex.mail.autouser.mail.SmtpService;
import ru.yandex.mail.autouser.tasks.DeleteStaleEmailsTask;
import ru.yandex.mail.autouser.tasks.ListSizeReportingTask;
import ru.yandex.mail.autouser.tasks.ReplyTask;
import ru.yandex.mail.autouser.tasks.RunCommandTask;
import ru.yandex.mail.autouser.utils.MessageTypes;

/**
 * @author Sergey Galyamichev
 */
@Component
public class CommandExecutor implements CommandVisitor {
    private static final Logger LOG = LoggerFactory.getLogger(CommandExecutor.class);
    private static final Map<String, ScheduledFuture> COMMANDS = new ConcurrentHashMap<>();

    private final ImapService imapService;
    private final SmtpService smtpService;
    private final ScheduledExecutorService scheduledExecutor;

    public CommandExecutor(ImapService imapService, SmtpService smtpService, ScheduledExecutorService scheduledExecutor) {
        this.imapService = imapService;
        this.smtpService = smtpService;
        this.scheduledExecutor = scheduledExecutor;
    }

    @Override
    public void visit(EmailListCommand command) {
        ListSizeReportingTask task = new ListSizeReportingTask(imapService, smtpService);
        COMMANDS.put(command.getId(),
                scheduledExecutor.scheduleAtFixedRate(task, 0, command.getRepeatInterval(), TimeUnit.MILLISECONDS));
    }

    @Override
    public void visit(DeleteAllCommand command) {
        LOG.debug("In visit(DeleteAllCommand command={})...", command);
        try {
            imapService.forEach(m -> true, ImapService::markDelete);
        } catch (Exception e) {
            LOG.error("Something wrong!", e);
        }
    }

    @Override
    public void visit(DeleteCommand command) {
        DeleteStaleEmailsTask task = new DeleteStaleEmailsTask(imapService, command.getStalePeriod());
        COMMANDS.put(command.getId(),
                scheduledExecutor.scheduleAtFixedRate(task, 0, command.getRepeatInterval(), TimeUnit.MILLISECONDS));
    }

    @Override
    public void visit(SendTextCommand command) {
        LOG.debug("In visit(SendTextCommand command={})...", command);
        smtpService.send(m -> prepareTextMessage(m, command));
        LOG.debug("Message sent...");

    }

    private void prepareTextMessage(MimeMessage mimeMessage, SendTextCommand command) throws Exception {
        MimeMessageHelper message = getMimeMessageHelper(mimeMessage, command.getAddressee());
        message.setText(command.getText());
        message.setSubject(command.getSubject());
    }

    private void prepareCommandMessage(MimeMessage mimeMessage, SendCommandCommand command) throws Exception {
        MimeMessageHelper message = getMimeMessageHelper(mimeMessage, command.getAddressee());
        message.setText(CommandConverter.toString(command.getSubCommand()));
        message.setSubject(MessageTypes.COMMAND.getPrettyName() + ' ' + command.getId());
    }

    private MimeMessageHelper getMimeMessageHelper(MimeMessage mimeMessage, String addressee) throws MessagingException, UnsupportedEncodingException {
        MimeMessageHelper message = new MimeMessageHelper(mimeMessage, true, StandardCharsets.UTF_8.name());
        smtpService.updateFrom(message);
        message.setTo(addressee);
        return message;
    }

    @Override
    public void visit(SendCommandCommand command) {
        LOG.debug("In visit(SendCommandCommand command={})...", command);
        smtpService.send(m -> prepareCommandMessage(m, command));
        LOG.debug("Message sent...");

    }

    @Override
    public void visit(ReplyToCommand command) {
        ReplyTask task = new ReplyTask(imapService, smtpService, command.getAddress());
        COMMANDS.put(command.getId(),
                scheduledExecutor.scheduleAtFixedRate(task, 0, command.getRepeatInterval(), TimeUnit.MILLISECONDS));
    }

    public boolean stop(String id) {
        ScheduledFuture task = COMMANDS.remove(id);
        return task.cancel(false);
    }

    @Override
    public void visit(StopCommand command) {
        stop(command.getTargetId());
    }

    public List<String> commandsList() {
        return COMMANDS.values().stream()
                .map(CommandConverter::toSafeString)
                .collect(Collectors.toList());
    }

    @Override
    public void visit(RepeatCommand command) {
        RunCommandTask task = new RunCommandTask(this, command.getSubCommand());
        COMMANDS.put(command.getId(),
                scheduledExecutor.scheduleAtFixedRate(task, 0, command.getRepeatInterval(), TimeUnit.MILLISECONDS));

    }
}
