package ru.yandex.mail.autouser.mail;

import java.util.Properties;

import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.URLName;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

/**
 * @author Sergey Galyamichev
 */
@Component
public class ImapService {
    private static final Logger LOG = LoggerFactory.getLogger(ImapService.class);
    public static final EmailFilter NOT_SEEN = m -> !m.isSet(Flags.Flag.SEEN);


    private Session session;
    private Store store;
    private Folder folder;

    private final String protocol = "imaps";
    private final String file = "INBOX";

    private final Properties properties;

    @Value("${yandex.username}")
    private String username;
    @Value("${yandex.password}")
    private String password;
    @Value("${yandex.imap.host}")
    private String host;
    @Value("${yandex.imap.port}")
    private String port;


    public ImapService(Properties imapJavaProperties) {
        properties = imapJavaProperties;
    }

    public boolean isLoggedIn() {
        return store.isConnected();
    }

    /**
     * to login to the mail host server
     */
    public void login() throws MessagingException {
        URLName url = new URLName(protocol, host, 993, file, username, password);
        session = Session.getInstance(properties, null);
        store = session.getStore(url);
        store.connect();
        folder = store.getFolder(url);
        folder.open(Folder.READ_WRITE);
    }

    public int expunge() throws MessagingException {
        login();
        try {
            return folder.expunge().length;
        } finally {
            logout();
        }
    }
    /**
     * to logout from the mail host server
     */
    public void logout() throws MessagingException {
        folder.close(true);
        store.close();
        store = null;
        session = null;
    }

    public int getMessageCount() {
        int messageCount = 0;
        try {
            messageCount = folder.getMessageCount();
        } catch (MessagingException me) {
            me.printStackTrace();
        }
        return messageCount;
    }

    public Message[] getMessages() throws MessagingException {
        return folder.getMessages();
    }

    public void forEach(EmailFilter filter, EmailProcessor processor) throws MessagingException {
        LOG.debug("Before login...");
        login();
        LOG.info("Logged in, messages to process: {}", folder.getMessageCount());
        try {
            for (Message message : getMessages()) {
                boolean isMessageProcessed = false;
                do {
                    try {
                        if (filter.test(message)) {
                            processor.onEmailReceived(message);
                        }
                        isMessageProcessed = true;
                    } catch (ArrayIndexOutOfBoundsException ignore) {
                        /* Doing this should force the internal messagesCache to get updated
                         * Unfortunately, this is also somewhat racy, depending on just how
                         * hard the mail folder is being hit */
                        try {
                            folder.getMessage(folder.getMessageCount());
                        } catch (ArrayIndexOutOfBoundsException ignored) {
                            /* There really isn't much you can do here, except try again.
                             * the good news is that this should hardly ever happen!!
                             * Good in this case is a relative term! */
                        }
                    }
                } while (!isMessageProcessed);
            }
            LOG.info("All messages processed");
        } finally {
            try {
                logout();
            } catch (Exception e) {
                LOG.warn("Exception on close", e);
            }
        }
        LOG.debug("Logged out");
    }

    public static void markDelete(Message message) throws MessagingException {
        message.setFlag(Flags.Flag.DELETED, true);
    }
}
