#include "common.h"

namespace NNodejs {

std::string toString(const Napi::Value& val) {
    return val.IsString() ? val.As<Napi::String>().Utf8Value() : std::string();
}

Expectation expectSize(size_t size, std::string_view msg) {
    return [=] (Napi::Env env, const Napi::CallbackInfo& info) {
        if (info.Length() != size) {
            Napi::TypeError::New(env, msg.data()).ThrowAsJavaScriptException();
            return true;
        }
        return false;
    };
}

Expectation expectSizes(size_t size1, size_t size2, size_t size3, std::string_view msg) {
    return [=] (Napi::Env env, const Napi::CallbackInfo& info) {
        if (info.Length() != size1 && info.Length() != size2 && info.Length() != size3) {
            Napi::TypeError::New(env, msg.data()).ThrowAsJavaScriptException();
            return true;
        }
        return false;
    };
}

Expectation expectSizes(size_t size1, size_t size2, std::string_view msg) {
    return [=] (Napi::Env env, const Napi::CallbackInfo& info) {
        if (info.Length() != size1 && info.Length() != size2) {
            Napi::TypeError::New(env, msg.data()).ThrowAsJavaScriptException();
            return true;
        }
        return false;
    };
}

Ensure::Ensure(const Napi::CallbackInfo& info, const std::vector<Expectation>& fns)
    : env(info.Env())
    , scope(env)
    , info(info)
    , failed(false)
{
    for (const auto& fn: fns) {
        if (!failed) {
            failed = fn(env, info);
        }
    }
}

Napi::Value Ensure::trycatch(std::function<Napi::Value(Napi::Env)> fn) const {
    try {
        if (failed) {
            return Napi::Value();
        }

        return fn(env);
    } catch(const std::exception& ex) {
        Napi::TypeError::New(env, ex.what()).ThrowAsJavaScriptException();
        return Napi::Value();
    }
}

Napi::Number box(Napi::Env env, int32_t val) {
    return Napi::Number::New(env, val);
}

Napi::Number box(Napi::Env env, uint32_t val) {
    return Napi::Number::New(env, val);
}

Napi::Number box(Napi::Env env, std::time_t val) {
    return Napi::Number::New(env, val);
}

Napi::Number box(Napi::Env env, double val) {
    return Napi::Number::New(env, val);
}

Napi::String box(Napi::Env env, std::string_view val) {
    return Napi::String::New(env, val.data());
}

Napi::String box(Napi::Env env, const std::string& val) {
    return Napi::String::New(env, val);
}

Napi::Object box(Napi::Env, const Napi::Object& val) {
    return val;
}

Napi::Boolean box(Napi::Env env, bool val) {
    return Napi::Boolean::New(env, val);
}

}
