#pragma once

#include <contrib/libs/node-addon-api/napi.h>
#include <string>
#include <ctime>
#include <map>


namespace NNodejs {

using Expectation = std::function<bool(Napi::Env, const Napi::CallbackInfo&)>;

template<class T>
Expectation expectArgIs(size_t position, std::string_view msg) {
    struct Type {
        static bool is(const Napi::Value& val, Napi::Object) {
            return val.IsObject();
        }
        static bool is(const Napi::Value& val, Napi::String) {
            return val.IsString();
        }
        static bool is(const Napi::Value& val, Napi::Number) {
            return val.IsNumber();
        }
        static bool is(const Napi::Value& val, Napi::Boolean) {
            return val.IsBoolean();
        }
    };

    return [=] (Napi::Env env, const Napi::CallbackInfo& info) {
        if (!Type::is(info[position], T())) {
            Napi::TypeError::New(env, msg.data()).ThrowAsJavaScriptException();
            return true;
        }
        return false;
    };
}

template<class T>
Expectation expectOptionalArgIs(size_t position, std::string_view msg) {
    return [=] (Napi::Env env, const Napi::CallbackInfo& info) {
        if (info.Length() >= position+1) {
            return expectArgIs<T>(position, msg)(env, info);
        }
        return false;
    };
}

Expectation expectSize(size_t size, std::string_view msg);
Expectation expectSizes(size_t size1, size_t size2, std::string_view msg);
Expectation expectSizes(size_t size1, size_t size2, size_t size3, std::string_view msg);

struct Ensure {
    Napi::Env env;
    Napi::HandleScope scope;
    const Napi::CallbackInfo& info;
    bool failed;

    Ensure(const Napi::CallbackInfo& info, const std::vector<Expectation>& fns);

    Napi::Value trycatch(std::function<Napi::Value(Napi::Env)> fn) const;
};

std::string toString(const Napi::Value& val);

Napi::Number box(Napi::Env env, int32_t val);
Napi::Number box(Napi::Env env, uint32_t val);
Napi::Number box(Napi::Env env, std::time_t val);
Napi::Number box(Napi::Env env, double val);
Napi::String box(Napi::Env env, std::string_view val);
Napi::String box(Napi::Env env, const std::string& val);
Napi::Object box(Napi::Env, const Napi::Object& val);
Napi::Boolean box(Napi::Env env, bool val);

template<class Key, class Value>
Napi::Object box(Napi::Env env, const std::map<Key, Value>& result) {
    Napi::Object obj = Napi::Object::New(env);

    for(const auto& [key, value]: result) {
        obj.Set(box(env, key), box(env, value));
    }

    return obj;
}

template<class Array>
Napi::Array box(Napi::Env env, const Array& array) {
    Napi::Array arr = Napi::Array::New(env, array.size());

    std::size_t i = 0;
    auto it = array.begin();
    auto end = array.end();
    while (it != end) {
        arr.Set(i++, box(env, *it++));
    }

    return arr;
}

}
