#include "convert.h"
#include <mail/nodejs/cpp/common/common.h>

#include <util/generic/typetraits.h>

namespace NNodejs {

using namespace NGeobase::NImpl;

Napi::Object box(Napi::Env env, const TGeolocation& geoloc) {
    Napi::Object location = Napi::Object::New(env);
    location["latitude"]  = box(env, geoloc.Location.Lat);
    location["longitude"] = box(env, geoloc.Location.Lon);

    Napi::Object obj = Napi::Object::New(env);
    obj["region_id"]            = box(env, geoloc.RegionId);
    obj["region_id_by_ip"]      = box(env, geoloc.RegionIdByIp);
    obj["region_id_by_gp"]      = box(env, geoloc.RegionIdByGp);
    obj["precision_by_ip"]      = box(env, geoloc.PrecisionByIp);
    obj["suspected_region_id"]  = box(env, geoloc.SuspectedRegionId);
    obj["precision"]            = box(env, geoloc.Precision);
    obj["point_id"]             = box(env, geoloc.PointId);
    obj["should_update_cookie"] = box(env, geoloc.ShouldUpdateCookie);
    obj["gid_is_trusted"]       = box(env, geoloc.GidIsTrusted);
    obj["location"]             = box(env, location);

    return obj;
}

Napi::Object box(Napi::Env env, const TIpTraits& ipTraits) {
    static const std::map<std::string, bool (TIpTraits::*)() const> TraitsFlagsCheckers {
          { "stub_range",   &NGeobase::NImpl::TIpTraits::IsStub }
        , { "reserved",     &NGeobase::NImpl::TIpTraits::IsReserved }
        , { "yandex_net",   &NGeobase::NImpl::TIpTraits::IsYandexNet }
        , { "yandex_staff", &NGeobase::NImpl::TIpTraits::IsYandexStaff }
        , { "yandex_turbo", &NGeobase::NImpl::TIpTraits::IsYandexTurbo }
        , { "tor",          &NGeobase::NImpl::TIpTraits::IsTor }
        , { "proxy",        &NGeobase::NImpl::TIpTraits::IsProxy }
        , { "vpn",          &NGeobase::NImpl::TIpTraits::IsVpn }
        , { "hosting",      &NGeobase::NImpl::TIpTraits::IsHosting }
        , { "mobile",       &NGeobase::NImpl::TIpTraits::IsMobile }
    };

    Napi::Object obj = Napi::Object::New(env);

    obj["region_id"] = box(env, ipTraits.RegionId);

    if (!ipTraits.AsnList.empty()) {
        obj["asn_list"] = box(env, ipTraits.AsnList);
    }

    if (!ipTraits.IspName.empty()) {
        obj["isp_name"] = box(env, ipTraits.IspName);
    }

    if (!ipTraits.OrgName.empty()) {
        obj["org_name"] = box(env, ipTraits.OrgName);
    }

    for (const auto& [name, value]: TraitsFlagsCheckers) {
        if ((ipTraits.*value)()) {
            obj[name] = box(env, true);
        }
    }

    return obj;
}

Napi::Object box(Napi::Env env, const TLinguistics& ling) {
    static const std::map<std::string, std::string TLinguistics::*> casesValuesMapping = {
        { "nominative",    &TLinguistics::NominativeCase },
        { "genitive",      &TLinguistics::GenitiveCase },
        { "dative",        &TLinguistics::DativeCase },
        { "prepositional", &TLinguistics::PrepositionalCase },
        { "preposition",   &TLinguistics::Preposition },
        { "locative",      &TLinguistics::LocativeCase },
        { "directional",   &TLinguistics::DirectionalCase },
        { "ablative",      &TLinguistics::AblativeCase },
        { "accusative",    &TLinguistics::AccusativeCase },
        { "instrumental",  &TLinguistics::InstrumentalCase },
    };

    Napi::Object obj = Napi::Object::New(env);

    for (const auto& [name, ptr]: casesValuesMapping) {
        obj[name] = box(env, ling.*ptr);
    }

    return obj;
}

Napi::Object box(Napi::Env env, const TRegion& region) {
    static TServiceGetter sg;

    Napi::Object obj = Napi::Object::New(env);

    for (const auto& fieldName: region.GetFieldsNames()) {
        switch (region.GetFieldType(fieldName)) {
            case RT::FieldType::TYPE_UINT:
                obj[fieldName] = box(env, region.GetUIntField(fieldName));
                break;
            case RT::FieldType::TYPE_INT:
                obj[fieldName] = box(env, region.GetIntField(fieldName));
                break;
            case RT::FieldType::TYPE_DOUBLE:
                obj[fieldName] = box(env, region.GetDoubleField(fieldName));
                break;
            case RT::FieldType::TYPE_BOOLEAN:
                obj[fieldName] = box(env, region.GetBoolField(fieldName));
                break;
            case RT::FieldType::TYPE_STRING:
                obj[fieldName] = box(env, region.GetStrField(fieldName));
                break;
            case RT::FieldType::TYPE_INT_LIST:
                obj[fieldName] = box(env, region.GetIntListField(fieldName));
                break;
        }
    }

    obj["services_names"] = box(env, sg.GetNamesByMask(region.GetServices()));

    return obj;
}

enum class FieldType {
    String,
    Boolean,
    Number
};

std::string toString(FieldType t) {
    switch (t) {
        case FieldType::String:  return "String";
        case FieldType::Boolean: return "Boolean";
        case FieldType::Number:  return "Number";
    }
}

template<FieldType fieldType, class Ptr>
void assign(NGeobase::NImpl::TLookup::TGeolocationInput& location, const Napi::Object& obj, std::string_view name, Ptr ptr, bool required) {
    Napi::Value field = obj.Get(name.data());
    if (!field.IsUndefined()) {
        bool expectedType;

        if constexpr (fieldType == FieldType::String) {
            expectedType = field.IsString();
        } else if constexpr (fieldType == FieldType::Boolean) {
            expectedType = field.IsBoolean();
        } else if constexpr (fieldType == FieldType::Number) {
            expectedType = field.IsNumber();
        } else {
            static_assert(TValueDependentFalse<fieldType>, "strange FieldType");
        }

        if (!expectedType) {
            throw std::invalid_argument(name.data() + std::string(" should be ") + toString(fieldType));
        }

        if constexpr (fieldType == FieldType::String) {
            location.*ptr = field.As<Napi::String>();
        } else if constexpr (fieldType == FieldType::Boolean) {
            location.*ptr = field.As<Napi::Boolean>();
        } else if constexpr (fieldType == FieldType::Number) {
            location.*ptr = field.As<Napi::Number>();
        } else {
            static_assert(TValueDependentFalse<fieldType>, "strange FieldType");
        }
    } else if (required) {
        throw std::invalid_argument(name.data() + std::string(" should be specified"));
    }
}

TLookup::TGeolocationInput toGeolocationInput(Napi::Env, const Napi::Object& obj) {
    TLookup::TGeolocationInput input;

    assign<FieldType::Number>(input, obj, "yandex_gid", &TLookup::TGeolocationInput::YandexGid, false);

    assign<FieldType::String>(input, obj, "ip",              &TLookup::TGeolocationInput::Ip,            true);
    assign<FieldType::String>(input, obj, "x_forwarded_for", &TLookup::TGeolocationInput::XForwardedFor, false);
    assign<FieldType::String>(input, obj, "x_real_ip",       &TLookup::TGeolocationInput::XRealIp,       false);
    assign<FieldType::String>(input, obj, "gpauto",          &TLookup::TGeolocationInput::GpAuto,        false);
    assign<FieldType::String>(input, obj, "override_point",  &TLookup::TGeolocationInput::OverridePoint, false);
    assign<FieldType::String>(input, obj, "user_points",     &TLookup::TGeolocationInput::UserPoints,    false);

    assign<FieldType::Boolean>(input, obj, "is_trusted",               &TLookup::TGeolocationInput::IsTrusted,              false);
    assign<FieldType::Boolean>(input, obj, "allow_yandex",             &TLookup::TGeolocationInput::AllowYandex,            false);
    assign<FieldType::Boolean>(input, obj, "disable_suspected_region", &TLookup::TGeolocationInput::DisableSuspectedRegion, false);

    return input;
}

Napi::Object box(Napi::Env env, const TTimezone& tzInfo) {
    Napi::Object obj = Napi::Object::New(env);
    obj["abbr"]   = box(env, tzInfo.Abbr);
    obj["dst"]    = box(env, tzInfo.Dst == "dst");
    obj["name"]   = box(env, tzInfo.Name);
    obj["offset"] = box(env, tzInfo.Offset);

    return obj;
}

}
