#include <contrib/libs/node-addon-api/napi.h>

#include <geobase/include/lookup.hpp>
#include <geobase/include/lookup_wrapper.hpp>
#include <geobase/include/service_getter.hpp>

#include <mail/nodejs/cpp/common/common.h>

#include "convert.h"

#include <type_traits>
#include <tuple>


using namespace NGeobase::NImpl;

namespace NNodejs {

struct GeobaseLookup: public Napi::ObjectWrap<GeobaseLookup> {
    static Napi::FunctionReference constructor;
    std::shared_ptr<TLookup> lookup;

    static Napi::Object Init(Napi::Env env, Napi::Object exports) {
        Napi::HandleScope scope(env);

        Napi::Function func = DefineClass(env, "GeobaseLookup", {
            InstanceMethod("getChildrenIds",          &GeobaseLookup::GetChildrenIds),
            InstanceMethod("getCountryRegion",        &GeobaseLookup::GetCountryRegion),
            InstanceMethod("getLinguistics",          &GeobaseLookup::GetLinguistics),
            InstanceMethod("getParentsIds",           &GeobaseLookup::GetParentsIds),
            InstanceMethod("getRegionById",           &GeobaseLookup::GetRegionById),
            InstanceMethod("getRegionByIp",           &GeobaseLookup::GetRegionByIp),
            InstanceMethod("getRegionIdByLocation",   &GeobaseLookup::GetRegionIdByLocation),
            InstanceMethod("getRegionsByType",        &GeobaseLookup::GetRegionsByType),
            InstanceMethod("getTimezoneById",         &GeobaseLookup::GetTimezoneById),
            InstanceMethod("getTraitsByIp",           &GeobaseLookup::GetTraitsByIp),
            InstanceMethod("isIdInRegion",            &GeobaseLookup::IsIdInRegion),
            InstanceMethod("isIpInRegion",            &GeobaseLookup::IsIpInRegion),
            InstanceMethod("makePinpointGeolocation", &GeobaseLookup::MakePinpointGeolocation),
            InstanceMethod("getIpTraits",             &GeobaseLookup::GetTraitsByIp),
        });

        constructor = Napi::Persistent(func);
        constructor.SuppressDestruct();

        exports.Set("GeobaseLookup", func);
        return exports;
    }

    GeobaseLookup(const Napi::CallbackInfo& info)
        : Napi::ObjectWrap<GeobaseLookup>(info)
    {
        Ensure(info, {
            expectSize(1, "Expect one string argument"),
            expectArgIs<Napi::String>(0, "First argument must be a string")
        }).trycatch([&](Napi::Env) {
            lookup = std::make_shared<TLookup>(info[0].As<Napi::String>().Utf8Value());
            return Napi::Value();
        });
    }

    Napi::Value GetChildrenIds(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(1, 2, "Bad arity - 1 argument required (id: int) + optional (domain: string)"),
            expectArgIs<Napi::Number>(0, "First argument must be a region-id (positive integer)")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->GetChildrenIds(
                info[0].As<Napi::Number>(),
                TLookupWrapper::GetCrimeaStatusByDomain(toString(info[1]))
            ));
        });
    }

    Napi::Value GetCountryRegion(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(1, 2, "Bad arity (id: int, [domain: string])"),
            expectArgIs<Napi::Number>(0, "First argument must be an int (region-id)")
        }).trycatch([&](Napi::Env env) {
            const auto country_id = lookup->GetCountryId(
                info[0].As<Napi::Number>(),
                TLookupWrapper::GetCrimeaStatusByDomain(toString(info[1]))
            );

            return box(env, lookup->GetRegionById(country_id));
        });
    }

    Napi::Value GetLinguistics(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(2, "Bad arity (region-id: int, lang-iso-code: string)"),
            expectArgIs<Napi::Number>(0, "First argument must be an int (region-id)"),
            expectArgIs<Napi::String>(1, "Second argument must be a string (lang-iso-code)")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->GetLinguistics(
                info[0].As<Napi::Number>(),
                info[1].As<Napi::String>().Utf8Value()
            ));
        });
    }

    Napi::Value GetParentsIds(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(1, 2, "Bad arity - 1 argument required (id: int) + optional (domain: string)"),
            expectArgIs<Napi::Number>(0, "First argument must be a region-id (positive integer)")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->GetParentsIds(
                info[0].As<Napi::Number>(),
                TLookupWrapper::GetCrimeaStatusByDomain(toString(info[1]))
            ));
        });
    }

    Napi::Value GetRegionById(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(1, 2, "Bad arity (id: int, [domain: string])"),
            expectArgIs<Napi::Number>(0, "First argument must be a region-id (positive integer)")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->GetRegionById(
                info[0].As<Napi::Number>(),
                TLookupWrapper::GetCrimeaStatusByDomain(toString(info[1]))
            ));
        });
    }

    Napi::Value GetRegionByIp(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(1, 2, "Bad arity (ip: string, [domain: string])"),
            expectArgIs<Napi::String>(0, "First argument must be a string")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->GetRegionByIp(
                info[0].As<Napi::String>(),
                TLookupWrapper::GetCrimeaStatusByDomain(toString(info[1]))
            ));
        });
    }

    Napi::Value GetRegionIdByLocation(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(2, 3, "Bad arity (lat: double, lon: double [, is_tune])"),
            expectArgIs<Napi::Number>(0, "1 arg must be double (lat)"),
            expectArgIs<Napi::Number>(1, "2 arg must be double (lon)"),
            expectOptionalArgIs<Napi::Boolean>(2, "3 arg must be boolean (is_tune)")
        }).trycatch([&](Napi::Env env) {
            Napi::Number lat = info[0].As<Napi::Number>();
            Napi::Number lon = info[1].As<Napi::Number>();

            Id id;
            if (info.Length() == 3) {
                id = lookup->GetRegionIdByLocation(lat.DoubleValue(), lon.DoubleValue(), info[2].As<Napi::Boolean>());
            } else {
                id = lookup->GetRegionIdByLocation(lat.DoubleValue(), lon.DoubleValue());
            }

            return box(env, id);
        });
    }

    Napi::Value GetRegionsByType(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(1, 2, "Bad arity (type: int, [domain: string])"),
            expectArgIs<Napi::Number>(0, "First argument must be a region type (positive integer)")
        }).trycatch([&](Napi::Env env) {
            const auto& regions = lookup->GetRegionsByType(
                info[0].As<Napi::Number>(),
                TLookupWrapper::GetCrimeaStatusByDomain(toString(info[1]))
            );

            Napi::Array array = Napi::Array::New(env, regions.size());
            for (std::size_t i = 0; i < regions.size(); i++) {
                array.Set(i, box(env, regions[i]));
            }

            return array;
        });
    }

    Napi::Value GetTimezoneById(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(1, "Bad arity (id: int)"),
            expectArgIs<Napi::Number>(0, "First argument must be a region-id (positive integer)")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->GetTimezoneById(info[0].As<Napi::Number>()));
        });
    }

    Napi::Value GetTraitsByIp(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(1, "Bad arity (ip: string)"),
            expectArgIs<Napi::String>(0, "First argument must be a string")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->GetTraitsByIp(info[0].As<Napi::String>()));
        });
    }

    Napi::Value IsIdInRegion(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(2, 3, "Bad arity - 2 arguments required (id: int, pid: int, [domain: string])"),
            expectArgIs<Napi::Number>(0, "First argument must be a region-id (positive integer)"),
            expectArgIs<Napi::Number>(1, "Second argument must be a region-id (positive integer)")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->IsIdInRegion(
                info[0].As<Napi::Number>(),
                info[1].As<Napi::Number>(),
                TLookupWrapper::GetCrimeaStatusByDomain(toString(info[1]))
            ));
        });
    }

    Napi::Value IsIpInRegion(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(2, 3, "Bad arity - 2 arguments required (ip: string, pid: int, [domain: string])"),
            expectArgIs<Napi::String>(0, "First argument must be a string"),
            expectArgIs<Napi::String>(1, "Second argument must be a region-id (positive integer)")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->IsIpInRegion(
                info[0].As<Napi::String>(),
                info[1].As<Napi::Number>(),
                TLookupWrapper::GetCrimeaStatusByDomain(toString(info[2]))
            ));
        });
    }

    Napi::Value MakePinpointGeolocation(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(3, "Bad arity (must be 3 args)"),
            expectArgIs<Napi::Object>(0, "1 argument must be a object (SearchData)"),
            expectArgIs<Napi::String>(1, "2 argument must be a string (YP)"),
            expectArgIs<Napi::String>(2, "3 argument must be a string (YS)")
        }).trycatch([&](Napi::Env env) {
            return box(env, lookup->MakePinpointGeolocation(
                toGeolocationInput(env, info[0].As<Napi::Object>()),
                info[1].As<Napi::String>().Utf8Value(),
                info[2].As<Napi::String>().Utf8Value()
            ));
        });
    }
};

Napi::FunctionReference GeobaseLookup::constructor;

Napi::Object InitGeobase(Napi::Env env, Napi::Object exports) {
    return GeobaseLookup::Init(env, exports);
}

NODE_API_MODULE(geobase, InitGeobase)

}
