#include <contrib/libs/node-addon-api/napi.h>

#include <langdetect/lookup.hpp>
#include <langdetect/domaininfo.hpp>
#include <langdetect/domain_filter.hpp>

#include "convert.h"

#include <mail/nodejs/cpp/common/common.h>

namespace NNodejs {

struct LangDetector: public Napi::ObjectWrap<LangDetector> {
    static Napi::FunctionReference constructor;
    std::shared_ptr<langdetect::lookup> lookup;

    static Napi::Object Init(Napi::Env env, Napi::Object exports) {
        Napi::HandleScope scope(env);

        Napi::Function func = DefineClass(env, "LangDetector", {
            InstanceMethod("find",              &LangDetector::Find),
            InstanceMethod("findWithoutDomain", &LangDetector::FindWithoutDomain),
            InstanceMethod("findDomain",        &LangDetector::FindDomain),
            InstanceMethod("list",              &LangDetector::List),
            InstanceMethod("cookie2language",   &LangDetector::Cookie2Language),
            InstanceMethod("language2cookie",   &LangDetector::Language2Cookie),
            InstanceMethod("trySwap",           &LangDetector::TrySwap),
        });

        constructor = Napi::Persistent(func);
        constructor.SuppressDestruct();

        exports.Set("LangDetector", func);
        return exports;
    }

    LangDetector(const Napi::CallbackInfo& info)
        : Napi::ObjectWrap<LangDetector>(info)
    {
        Ensure(info, {
            expectSize(1, "Expect one argument: path to config-file"),
            expectArgIs<Napi::String>(0, "First argument must be a string")
        }).trycatch([&](Napi::Env) {
            lookup = std::make_shared<langdetect::lookup>(info[0].As<Napi::String>().Utf8Value());
            return Napi::Value();
        });
    }

    Napi::Value Find(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(1, "expect exactly one argument"),
            expectArgIs<Napi::Object>(0, "first argument must be on object {'domain': 'http://mail.yandex.ru/neo2', 'filter': 'tt,ru,uk', 'geo': '24896,20529,20524,187,166,10001,10000'}")
        }).trycatch([&](Napi::Env env) {
            langdetect::langinfo langInfo;
            FindParams params(info[0].As<Napi::Object>());

            lookup->find(params.userInfo, langInfo, &params.filter, params.defaultLanguage);

            return box(env, langInfo);
        });
    }

    Napi::Value FindWithoutDomain(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(1, "expect exactly one argument"),
            expectArgIs<Napi::Object>(0, "first argument must be on object {'domain': 'http://mail.yandex.ru/neo2', 'filter': 'tt,ru,uk', 'geo': '24896,20529,20524,187,166,10001,10000'}")
        }).trycatch([&](Napi::Env env) {
            langdetect::langinfo langInfo;
            FindParams params(info[0].As<Napi::Object>());

            lookup->find_without_domain(params.userInfo, langInfo, &params.filter, params.defaultLanguage);

            return box(env, langInfo);
        });
    }

    Napi::Value FindDomain(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSizes(3, 4, ""),
            expectArgIs<Napi::String>(0, "geo list must be a string"),
            expectArgIs<Napi::String>(1, "domain filter must be a string"),
            expectArgIs<Napi::String>(2, "host must be a string"),
            expectOptionalArgIs<Napi::String>(3, "4 arg must be string")
        }).trycatch([&](Napi::Env env) -> Napi::Value {
            langdetect::domaininfo domainInfo;
            langdetect::domain_filter domainFilter;

            if (!domainInfo.parse_geo_regions(toString(info[0]))) {
                throw std::runtime_error("Invalid geo list");
            }

            domainFilter.add(toString(info[1]));
            if (domainFilter.empty()) {
                throw std::runtime_error("Strange domain filter");
            }

            if (!domainInfo.parse_host(toString(info[2]))) {
                throw std::runtime_error("Strange host");
            }

            if (info.Length() == 4) {
                if (!domainInfo.parse_cookie_cr(toString(info[3]))) {
                    throw std::runtime_error("Invalid cookie-cr");
                }
            }

            langdetect::find_domain_result result;
            lookup->find_domain_ex(domainInfo, &domainFilter, result);
            if (result.domain.empty()) {
                return env.Null();
            }

            return box(env, result);
        });
    }

    Napi::Value List(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(1, "expected only one arg"),
            expectArgIs<Napi::Object>(0, "Object expected")
        }).trycatch([&](Napi::Env env) -> Napi::Value {
            FindParams params(info[0].As<Napi::Object>());

            langdetect::lookup::lang_list_type list;
            if (!lookup->list(params.userInfo, list, params.filter.empty() ? nullptr : &params.filter, params.defaultLanguage)) {
                return env.Null();
            }

            if (list.empty()) {
                return env.Null();
            }

            return box(env, list);
        });
    }

    Napi::Value Cookie2Language(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(1, "expected only one arg"),
            expectArgIs<Napi::Number>(0, "Number expected")
        }).trycatch([&](Napi::Env env) -> Napi::Value {
            int32_t cookieValue = info[0].ToNumber().Int32Value();
            if (cookieValue > 0) {
                langdetect::langinfo langInfo;
                lookup->get_info(cookieValue, langInfo);

                if (!langInfo.code.empty()) {
                    return box(env, langInfo.code);
                }
            }
            return env.Null();
        });
    }

    Napi::Value Language2Cookie(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(1, ""),
            expectArgIs<Napi::String>(0, "Bad arity, first argument must be string (lang)")
        }).trycatch([&](Napi::Env env) -> Napi::Value {
            std::string langString = toString(info[0]);

            if (langString.size()) {
                langdetect::langinfo langInfo;
                lookup->get_info(langString, langInfo);

                if (!langInfo.code.empty()) {
                    return box(env, langInfo.cookie_value);
                }
            }
            return env.Null();
        });
    }

    Napi::Value TrySwap(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(0, "")
        }).trycatch([&](Napi::Env env) {
            if (lookup->need_swap()) {
                lookup.reset(new langdetect::lookup(lookup->filepath()));
                return box(env, true);
            } else {
                return box(env, false);
            }
        });
    }
};

Napi::FunctionReference LangDetector::constructor;

Napi::Object InitDetector(Napi::Env env, Napi::Object exports) {
    return LangDetector::Init(env, exports);
}

NODE_API_MODULE(langdetect, InitDetector)

}
