#include <contrib/libs/node-addon-api/napi.h>

#include <uatraits/detector.hpp>

#include <boost/algorithm/string.hpp>

#include <mail/nodejs/cpp/common/common.h>


namespace NNodejs {

using ResultType = uatraits::detector::result_type;

struct Detector: public Napi::ObjectWrap<Detector> {
    static Napi::FunctionReference constructor;
    std::shared_ptr<uatraits::detector> detector;

    static Napi::Object Init(Napi::Env env, Napi::Object exports) {
        Napi::HandleScope scope(env);

        Napi::Function func = DefineClass(env, "Detector", {
            InstanceMethod("detect",          &Detector::Detect),
            InstanceMethod("detectByHeaders", &Detector::DetectByHeaders),
        });

        constructor = Napi::Persistent(func);
        constructor.SuppressDestruct();

        exports.Set("Detector", func);
        return exports;
    }

    Detector(const Napi::CallbackInfo& info)
        : Napi::ObjectWrap<Detector>(info)
    {
        Ensure(info, {
            expectSizes(1, 2, 3, "Expect one, two or three arguments"),
            expectArgIs<Napi::String>(0, "first argument must be a string"),
            expectOptionalArgIs<Napi::String>(1, "second argument must be a string"),
            expectOptionalArgIs<Napi::String>(2, "third argument must be a string")
        }).trycatch([&](Napi::Env) {
            std::string path = info[0].As<Napi::String>().Utf8Value();

            if (info.Length() == 1) {
                detector = std::make_shared<uatraits::detector>(path.c_str());
            } else {
                std::string profilesPath = info[1].As<Napi::String>().Utf8Value();

                if (info.Length() == 2) {
                    detector = std::make_shared<uatraits::detector>(
                        path.c_str(), profilesPath.c_str()
                    );
                } else {
                    std::string extraPath = info[2].As<Napi::String>().Utf8Value();
                    detector = std::make_shared<uatraits::detector>(
                        path.c_str(), profilesPath.c_str(), extraPath.c_str()
                    );
                }
            }

            return Napi::Value();
        });
    }

    Napi::Value Detect(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(1, ""),
            expectArgIs<Napi::String>(0, "")
        }).trycatch([&](Napi::Env env) {
            return box(
                env, detector->detect(info[0].As<Napi::String>().Utf8Value())
            );
        });
    }

    Napi::Value DetectByHeaders(const Napi::CallbackInfo& info) {
        return Ensure(info, {
            expectSize(1, ""),
            expectArgIs<Napi::Object>(0, "")
        }).trycatch([&](Napi::Env env) {
            ResultType headers;
            ResultType result;

            Napi::Object jsObject = info[0].As<Napi::Object>();
            Napi::Array headerNames = jsObject.GetPropertyNames();
            for (uint32_t index = 0; index < headerNames.Length(); index++) {
                Napi::String headerName = headerNames.Get(index).As<Napi::String>();
                Napi::String headerValue = jsObject.Get(headerName).As<Napi::String>();

                if (headerName.IsString() && headerValue.IsString()) {
                    headers[boost::to_lower_copy(headerName.Utf8Value())] = headerValue.Utf8Value();
                }
            }

            detector->detect(headers, result);

            return box(env, result);
        });
    }
};

Napi::FunctionReference Detector::constructor;

Napi::Object InitUatraits(Napi::Env env, Napi::Object exports) {
    return Detector::Init(env, exports);
}

NODE_API_MODULE(uatraits, InitUatraits)

}
