import json
import os.path

from email.mime.text import MIMEText
from library.python import resource
from smtplib import LMTP, SMTPResponseException

from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import BaseComponent
from mail.devpack.lib.components.pyremock import PyremockComponent
from mail.devpack.lib.components.fakebb import FakeBlackbox
from mail.devpack.lib.components.tvmapi import TvmApi
from mail.devpack.lib.yhttp_service import YplatformHttpService

from mail.furita.devpack.components.furita import FuritaComponent as Furita
from mail.hound.devpack.components.application import Hound
from mail.mdbsave.devpack.components.mdbsave import MdbSaveComponent as MdbSave
from mail.notsolitesrv.devpack.components.relay import RelayComponent


class NslsBase(BaseComponent):
    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(NslsBase, cls).gen_config(port_generator, config=config)
        http = YplatformHttpService.gen_config(port_generator, config=config)
        return {
            **base,
            **http,
            'smtpserver_port': next(port_generator)
        }

    def __init__(self, env, components):
        super(NslsBase, self).__init__(env, components)
        self.yhttp = YplatformHttpService(env, self.NAME, binary_name='notsolitesrv', custom_path='notsolitesrv')
        self._smtpserver_host = 'localhost'
        self._smtpserver_port = self.config[self.name]['smtpserver_port']

    def init_root(self):
        self.yhttp.init_root()

        helpers.mkdir_recursive(self.app_config_path)

        self.write_resource('notsolitesrv/base.yml', os.path.join(self.app_config_path, 'base.yml'))
        self.write_resource('notsolitesrv/tvm_secret', os.path.join(self.secrets_path, 'tvm_secret'))
        self.write_resource('notsolitesrv/merge_rules.json', os.path.join(self.app_config_path, 'merge_rules.json'))
        self.write_resource(
            'notsolitesrv/trivial_subjects.json',
            os.path.join(self.app_config_path, 'trivial_subjects.json'))
        self.write_resource(
            'notsolitesrv/subscription.rules.xml',
            os.path.join(self.app_config_path, 'subscription.rules.xml'))
        self.write_resource(
            'notsolitesrv/firstline.rules.txt',
            os.path.join(self.app_config_path, 'firstline.rules.txt'))
        self.write_resource('notsolitesrv/ssl.pem', os.path.join(self.secrets_path, 'ssl.pem'))

        resources_path = os.path.join(self.get_root(), 'app', 'resources')
        helpers.mkdir_recursive(resources_path)
        self.write_resource('recognizer/dict.dict', os.path.join(resources_path, 'dict.dict'))
        self.write_resource('recognizer/queryrec.dict', os.path.join(resources_path, 'queryrec.dict'))
        self.write_resource('recognizer/queryrec.weights', os.path.join(resources_path, 'queryrec.weights'))

        open(os.path.expanduser(os.path.join(resources_path, 'related_uids')), 'a')

        self._init_root()

    def _init_root(self):
        pass

    def write_resource(self, resource_path, destination_path):
        helpers.write2file(what=resource.find(resource_path), path=destination_path)

    @property
    def app_path(self):
        return self.yhttp.get_app_path()

    @property
    def app_config_path(self):
        return os.path.join(self.app_path, 'config')

    @property
    def app_config_file(self):
        return os.path.join(self.app_config_path, 'config.yml')

    @property
    def secrets_path(self):
        return self.yhttp.get_secrets_path()

    def get_root(self):
        return self.yhttp.get_root()

    def smtpserver_host(self):
        return self._smtpserver_host

    def smtpserver_port(self):
        return self._smtpserver_port

    def webserver_port(self):
        return self.yhttp.webserver_port

    def start(self):
        self.yhttp.start('pong', config_path=self.app_config_file)

    def stop(self):
        self.yhttp.stop()

    def info(self):
        info = self.yhttp.info()
        return {
            **info,
            'state': self.state
        }

    def ping(self):
        return self.yhttp.ping()


class NslsCommon(NslsBase):
    DEPS = [PyremockComponent]

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(NslsCommon, cls).gen_config(port_generator, config=config)
        return base

    def __init__(self, env, components):
        super(NslsCommon, self).__init__(env, components)

    def pyremock_port(self):
        return self.components[PyremockComponent].port

    def send_message(self, msg: MIMEText):
        result = dict()
        try:
            client = LMTP(host=self.smtpserver_host(), port=self.smtpserver_port())
            client.send_message(msg)
            result["all"] = (250, b"Ok")
            client.quit()
        except SMTPResponseException as exc:
            result["all"] = (exc.smtp_code, exc.smtp_error)
        return result

    def send_http_message(self, msg):
        return self.yhttp.post(target="/store", data=json.dumps(msg))


class NslsComponent(NslsCommon):
    NAME = 'nsls'
    DEPS = [PyremockComponent, TvmApi, RelayComponent]

    def __init__(self, env, components):
        super(NslsComponent, self).__init__(env, components)

    def _init_root(self):
        devpack_config = self.yhttp.format_config(
            config=resource.find('notsolitesrv/devpack.yml'),
            tvm_api_port=self.components[TvmApi].port,
            relay_port=self.components[RelayComponent].port,
            pyremock_port=self.pyremock_port(),
            smtpserver_port=self.smtpserver_port()
        )
        helpers.write2file(what=devpack_config, path=self.app_config_file)


class NslsLocalComponent(NslsCommon):
    NAME = 'nsls_local'
    DEPS = [PyremockComponent, FakeBlackbox, Furita, Hound, MdbSave, TvmApi, RelayComponent]

    def make_config(self, params={}):
        default_params = dict(
            tvm_api_port=self.components[TvmApi].port,
            relay_port=self.components[RelayComponent].port,
            furita_port=self.components[Furita].webserver_port(),
            mdbsave_port=self.components[MdbSave].webserver_port(),
            blackbox_port=self.components[FakeBlackbox].port,
            pyremock_port=self.pyremock_port(),
            smtpserver_port=self.smtpserver_port(),
        )
        kwargs = {**default_params, **params}
        return self.yhttp.format_config(
            config=resource.find('notsolitesrv/local.yml'),
            **kwargs
        )

    def __init__(self, env, components):
        super(NslsLocalComponent, self).__init__(env, components)

    def _init_root(self):
        devpack_config = self.make_config()
        helpers.write2file(what=devpack_config, path=self.app_config_file)


class NslsTesting(NslsBase):
    NAME = 'nsls_testing'
    DEPS = []

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(NslsTesting, cls).gen_config(port_generator, config=config)
        return base

    def __init__(self, env, components):
        super(NslsTesting, self).__init__(env, components)

    def _init_root(self):
        devpack_config = self.yhttp.format_config(
            config=resource.find('notsolitesrv/testing.yml'),
            smtpserver_port=self.smtpserver_port()
        )
        helpers.write2file(what=devpack_config, path=self.app_config_file)
