from aiosmtpd import controller
from aiosmtpd.smtp import Envelope
from email import message_from_bytes
from smtplib import SMTP

import backoff

from mail.devpack.lib.components.base import BaseComponent
from mail.devpack.lib.errors import DevpackError


class Message:
    def __init__(self, envelope: Envelope):
        self.envelope = envelope
        self.mime = message_from_bytes(envelope.content)
        self.msg_id = self.mime["Message-Id"]


class MessagesStorage:
    def __init__(self):
        self.__msgs = dict()

    def get_messages(self, msg_id):
        return self.__msgs.get(msg_id, None)

    def get_messages_by_rcpt(self, rcpt):
        return [
            msg
            for key in self.__msgs
            for msg in self.__msgs[key]
            if rcpt in msg.envelope.rcpt_tos
        ]

    def prepare_message(self, _, envelope) -> Message:
        return Message(envelope)

    async def handle_DATA(self, _, session, envelope):
        message = self.prepare_message(session, envelope)
        if message.msg_id not in self.__msgs:
            self.__msgs[message.msg_id] = []
        self.__msgs[message.msg_id].append(message)
        return "250 OK"


class SmtpServer:
    def __init__(self, host, port, handler):
        self.__server = controller.Controller(hostname=host, port=port, handler=handler)

    def start(self):
        self.__server.start()

    def stop(self):
        self.__server.stop()


class RelayComponent(BaseComponent):
    NAME = 'relay'

    @staticmethod
    def gen_config(port_generator, config=None):
        return {"relay_port": next(port_generator)}

    def __init__(self, env, components):
        super(RelayComponent, self).__init__(env, components)
        self.__storage = MessagesStorage()
        self.__relay_host = 'localhost'
        self.__relay_port = self.config[self.name]['relay_port']
        self.__relay_server = SmtpServer(host=self.__relay_host, port=self.__relay_port, handler=self.__storage)

    def start(self):
        self.logger.info('starting %s on port %d', self.NAME, self.port)
        self.__relay_server.start()
        try:
            self.wait_successful_connect()
        except Exception as e:
            self.logger.exception(e)
            raise DevpackError("%s does not respond to / after start" % self.NAME)
        self.logger.info('%s started', self.NAME)

    def stop(self):
        self.logger.info('stopping %s on port %d', self.NAME, self.port)
        self.__relay_server.stop()
        self.logger.info('%s stopped', self.NAME)

    @property
    def host(self):
        return self.__relay_host

    @property
    def port(self):
        return self.__relay_port

    @property
    def storage(self):
        return self.__storage

    def ping(self):
        return SMTP().connect(host=self.host, port=self.port)

    @backoff.on_exception(
        backoff.constant,
        Exception,
        max_tries=10,
        interval=2
    )
    def wait_successful_connect(self):
        response = self.ping()
        self.logger.info("connect to relay code=%d, message='%s'", response[0], response[1])
        if response[0] == 220:
            self.logger.info('successfully connected to relay, ok')
        else:
            raise DevpackError("unable to connect to relay")
