#pragma once

#include "pcre.h"
#include "subscription_source.h"

#include <contrib/libs/expat/expat.h>

#include <string>
#include <memory>
#include <vector>

namespace NNotSoLiteSrv::NFirstline::NLib {


class MyElement;


struct MyData {
    SubscriptionSource* v;
    MyElement* currEl;
};


class MyElement {
public:
    MyElement()
        : parent_(0)
        , badcounter_(0)
    {
        ;
    }
    virtual ~MyElement() {}

    auto begin() const {
        return els_.begin();
    }

    auto end() const {
        return els_.end();
    }

    void push_back(MyElement* el) {
        els_.emplace_back(el);
    }

    void start(MyData* data, const char* name, const char** atts) {
        if (badcounter_ > 0) {
            badcounter_++;
        } else {
            if (MyElement* el = create(data, name, atts)) {
                el->parent_ = this;
                push_back(el);
                data->currEl = el;
            } else {
                badcounter_ = 1;
            }
        }
    }

    void end(MyData* data) {
        handleCharacterData(ch_data_);
        ch_data_.clear();
        if (badcounter_ > 0) {
            badcounter_--;
        } else {
            data->currEl = parent_;
        }
    }

    void character_data(const XML_Char* _str, int len) {
        ch_data_.append(_str, len);
    }

    virtual void handleCharacterData(std::string& /*str*/) {}

protected:
    virtual MyElement* create(MyData* /*data*/, const char* /*name*/, const char** /*atts*/) {
        return 0;
    }

private:
    MyElement* parent_;
    int badcounter_;
    std::string ch_data_;

    std::vector<std::unique_ptr<MyElement>> els_;
};

class MatchEl : public MyElement {
public:
    MatchEl(MyData* data, Match* match, const char** atts) {
        for (; atts && *atts; atts += 2) {
            if (!strcasecmp(atts[0], "regex")) {
                match->regex_ = atts[1];
                break;
            }
        }
        match->compile(data->v->tables);
    }
};


class FromEl : public MyElement {
public:
    FromEl(From* from, const char** atts) : from_(from) {
        for (; atts && *atts; atts += 2) {
            const char* val = atts[1];
            if (!strcasecmp(atts[0], "email")) {
                from_->email = val;
            }
        }
    }

    MyElement* create(MyData* data, const char* name, const char** atts) {
        if (!strcasecmp(name, "match")) {
            Match* match = new Match;
            from_->match.reset(match);
            return new MatchEl(data, match, atts);
        }
        return 0;
    }

private:
    From* from_;
};


using SmtpFromEl = FromEl;


class SubjectEl : public MyElement {
public:
    SubjectEl(Subject* subj) : subj_(subj) {}

    MyElement* create(MyData* data, const char* name, const char** atts) {
        if (!strcasecmp(name, "match")) {
            Match* match = new Match;
            subj_->match.reset(match);
            return new MatchEl(data, match, atts);
        }
        return 0;
    }
private:
    Subject* subj_;
};


class RuleInserter {
public:
    virtual ~RuleInserter() {}
    virtual void insert(Rule*) = 0;
};


class RuleInserter0: public RuleInserter {
public:
    explicit RuleInserter0(std::unique_ptr<Rule>& rule)
        : rule_(rule)
    {
        ;
    }
    void insert(Rule* rule) {
        rule_.reset(rule);
    }
private:
    std::unique_ptr<Rule>& rule_;
};


class RuleInserter1: public RuleInserter {
public:
    explicit RuleInserter1(Rules& rules)
        : rules_(rules)
    {
        ;
    }

    void insert(Rule* rule) {
        rules_.emplace_back(rule);
    }

private:
    Rules& rules_;
};


class Rule_rEl: public MyElement {
public:
    Rule_rEl(MyData* data, Rule_r* rule, const char** atts) {
        for (; atts && *atts; atts += 2) {
            std::string val(atts[1]);
            if (!strcasecmp(atts[0], "regex")) {
                rule->regex_.swap(val);
            } else if (!strcasecmp(atts[0], "result")) {
                rule->result_.swap(val);
            }
        }
        rule->compile(data->v->tables);
    }
};


class Rule_srEl: public MyElement {
public:
    Rule_srEl(MyData* data, Rule_sr* rule, const char** atts) {
        for (; atts && *atts; atts += 2) {
            std::string val(atts[1]);
            if (!strcasecmp(atts[0], "regex")) {
                rule->regex_.swap(val);
            } else if (!strcasecmp(atts[0], "include_match")) {
                rule->includeMatch_ = ((!strcasecmp(val.c_str(), "true")) ? true : false);
            }
        }
        rule->compile(data->v->tables);
    }
};


class Rule_prEl: public MyElement {
public:
    Rule_prEl(MyData* data, Rule_pr* rule, const char** atts) {
        for (; atts && *atts; atts += 2) {
            std::string val(atts[1]);
            if (!strcasecmp(atts[0], "regex")) {
                rule->regex_.swap(val);
            } else if (!strcasecmp(atts[0], "include_match")) {
                rule->includeMatch_ = ((!strcasecmp(val.c_str(), "true")) ? true : false);
            }
        }
        rule->compile(data->v->tables);
    }
};


class Rule_erEl: public MyElement {
public:
    Rule_erEl(MyData* data, Rule_er* rule, const char** atts) {
        for (; atts && *atts; atts += 2) {
            std::string val(atts[1]);
            if (!strcasecmp(atts[0], "regex")) {
                rule->regex_.swap(val);
            } else if (!strcasecmp(atts[0], "glob")) {
                rule->glob_ = ((!strcasecmp(val.c_str(), "true")) ? true : false);
            }
        }
        rule->compile(data->v->tables);
    }
};


class Rule_slEl: public MyElement {
public:
    Rule_slEl(MyData*, Rule_sl* rule, const char** atts) {
        for (; atts && *atts; atts += 2) {
            const char* val = atts[1];
            if (!strcasecmp(atts[0], "lines")) {
                rule->lines_ = atoi(val);
            }
        }
    }
};


class Rule_plEl: public MyElement {
public:
    Rule_plEl(MyData*, Rule_pl* rule, const char** atts) {
        for (; atts && *atts; atts += 2) {
            const char* val = atts[1];
            if (!strcasecmp(atts[0], "lines")) {
                rule->lines_ = atoi(val);
            }
        }
    }
};


class IntEl: public MyElement {
public:
    IntEl(int* i)
        : i_(i)
    {
        ;
    }

    void handleCharacterData(std::string& str) {
        *i_ = atoi(str.c_str());
    }

private:
    int* i_;
};


class StringEl: public MyElement {
public:
    StringEl(std::string* label)
        : label_(label)
    {
        ;
    }
    void handleCharacterData(std::string& str) {
        label_->swap(str);
    }

private:
    std::string* label_;
};


class RuleEl: public MyElement {
public:
    RuleEl(RuleInserter* inserter)
        : inserter_(inserter)
    {
        ;
    }
    MyElement* create(MyData* data, const char* name, const char** atts);

private:
    std::unique_ptr<RuleInserter> inserter_;
};


class RuleRestrictionEl: public MyElement {
public:
    explicit RuleRestrictionEl(RuleRestriction* rule)
        : rr_(rule)
    {
        if (rr_) {
            rule_.reset(new RuleEl(new RuleInserter0(rr_->rule_)));
        }
    }

    MyElement* create(MyData* data, const char* name, const char** atts) {
        if (!rr_ || !rule_.get()) {
            return 0;
        }
        if (!strcasecmp(name, "subject")) {
            Subject* subj = new Subject;
            rr_->subject_.reset(subj);
            return new SubjectEl(subj);
        }
        return rule_->create(data, name, atts);
    }
private:
    RuleRestriction* rr_;
    std::unique_ptr<RuleEl> rule_;
};


MyElement* RuleEl::create(MyData* data, const char* name, const char** atts) {
    if (!inserter_.get()) {
        return 0;
    }
    if (!strcasecmp(name, "rule_r")) {
        Rule_r* rule = new Rule_r;
        inserter_->insert(rule);
        return new Rule_rEl(data, rule, atts);
    } else if (!strcasecmp(name, "rule_sr")) {
        Rule_sr* rule = new Rule_sr;
        inserter_->insert(rule);
        return new Rule_srEl(data, rule, atts);
    } else if (!strcasecmp(name, "rule_pr")) {
        Rule_pr* rule = new Rule_pr;
        inserter_->insert(rule);
        return new Rule_prEl(data, rule, atts);
    } else if (!strcasecmp(name, "rule_er")) {
        Rule_er* rule = new Rule_er;
        inserter_->insert(rule);
        return new Rule_erEl(data, rule, atts);
    } else if (!strcasecmp(name, "rule_sl")) {
        Rule_sl* rule = new Rule_sl;
        inserter_->insert(rule);
        return new Rule_slEl(data, rule, atts);
    } else if (!strcasecmp(name, "rule_pl")) {
        Rule_pl* rule = new Rule_pl;
        inserter_->insert(rule);
        return new Rule_plEl(data, rule, atts);
    } else if (!strcasecmp(name, "a")) {
        RuleAlternation* rule = new RuleAlternation;
        inserter_->insert(rule);
        return new RuleEl(new RuleInserter1(rule->rules));
    } else if (!strcasecmp(name, "c")) {
        RuleComposition* rule = new RuleComposition;
        inserter_->insert(rule);
        return new RuleEl(new RuleInserter1(rule->rules));
    } else if (!strcasecmp(name, "i")) {
        RuleIdentity* rule = new RuleIdentity;
        inserter_->insert(rule);
        return new RuleEl(0);
    } else if (!strcasecmp(name, "r")) {
        RuleRestriction* rule = new RuleRestriction;
        inserter_->insert(rule);
        return new RuleRestrictionEl(rule);
    }
    return 0;
}


class FlEl: public MyElement {
public:
    FlEl(RuleInserter* text, RuleInserter* html, const char** atts) {
        for (; atts && *atts; atts += 2) {
            if (!strcasecmp(atts[0], "input_type")) {
                if (!strcasecmp(atts[1], "text")) {
                    rule_.reset(new RuleEl(text));
                    if (html) {
                        delete html;
                        html = 0;
                    }
                    break;
                } else if (!strcasecmp(atts[1], "html")) {
                    rule_.reset(new RuleEl(html));
                    if (text) {
                        delete text;
                        text = 0;
                    }
                    break;
                }
            }
        }
    }

    MyElement* create(MyData* data, const char* name, const char** atts) {
        if (!rule_.get()) {
            return 0;
        }
        return rule_->create(data, name, atts);
    }

private:
    std::unique_ptr<RuleEl> rule_;
};


class SubscriptionEl: public MyElement {
public:
    explicit SubscriptionEl(SubscriptionHelper* s)
        : s_(s)
    {
        ;
    }

    MyElement* create(MyData* /*data*/, const char* name, const char** atts) {
        if (!strcasecmp(name, "from")) {
            From* from = new From;
            s_->froms.emplace_back(from);
            return new FromEl(from, atts);
        } else if (!strcasecmp(name, "smtpfrom")) {
            SmtpFrom* from = new SmtpFrom;
            s_->smtpfroms.emplace_back(from);
            return new SmtpFromEl(from, atts);
        } else if (!strcasecmp(name, "subject")) {
            Subject* subj = new Subject;
            s_->subject.reset(subj);
            return new SubjectEl(subj);
        } else if (!strcasecmp(name, "url")) {
            return new RuleEl(new RuleInserter0(s_->url));
        } else if (!strcasecmp(name, "label")) {
            std::string* label = new std::string;
            s_->labels.emplace_back(label);
            return new StringEl(label);
        } else if (!strcasecmp(name, "auth")) {
            return new IntEl(&s_->auth);
        } else if (!strcasecmp(name, "fl")) {
            return new FlEl(new RuleInserter0(s_->text_fl), new RuleInserter0(s_->html_fl), atts);
        } else if (!strcasecmp(name, "service")) {
            std::string* s = new std::string;
            s_->service.reset(s);
            return new StringEl(s);
        }
        return 0;
    }

private:
    SubscriptionHelper* s_;
};


class PcreSettingsEl: public MyElement {
public:
    explicit PcreSettingsEl(PcreSettings* pcre)
        : pcre_(pcre)
    {
        ;
    }

    MyElement* create(MyData* /*data*/, const char* name, const char** /*atts*/) {
        if (!strcasecmp(name, "match_limit_recursion")) {
            return new IntEl(&pcre_->matchLimitRecursion);
        }
        return 0;
    }

private:
    PcreSettings* pcre_;
};


class SubscriptionSourceEl: public MyElement {
public:
    explicit SubscriptionSourceEl(SubscriptionSource* v)
        : v_(v)
    {
        ;
    }

    MyElement* create(MyData* /*data*/, const char* name, const char** /*atts*/) {
        if (!strcasecmp(name, "subscription")) {
            SubscriptionHelper* s = new SubscriptionHelper;
            v_->push_back(s);
            return new SubscriptionEl(s);
        } else if (!strcasecmp(name, "pcre_settings")) {
            PcreSettings* pcre = new PcreSettings;
            v_->pcre.reset(pcre);
            g_pcre_settings = pcre;
            return new PcreSettingsEl(pcre);
        }
        return 0;
    }

private:
    SubscriptionSource* v_;
};


class RootEl: public MyElement {
public:
    MyElement* create(MyData* data, const char* name, const char** /*atts*/) {
        if (!strcasecmp(name, "subscriptions")) {
            return new SubscriptionSourceEl(data->v);
        }
        return 0;
    }
};


} // namespace NNotSoLiteSrv::NFirstline::NLib
