#pragma once

#include "message_header_info.h"
#include "pcre.h"
#include "subject.h"

#include <boost/utility/string_ref.hpp>

#include <string>
#include <memory>
#include <array>
#include <vector>

namespace NNotSoLiteSrv::NFirstline::NLib {

class Rule {
public:
    virtual ~Rule() {}
    virtual int execute(const std::string& src, std::string& dst, const MessageHeaderInfo& hi) const = 0;
};


using Rules = std::vector<std::unique_ptr<Rule>>;


class RuleComposition: public Rule {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo& hi) const override {
        dst.clear();
        std::string asrc = src;

        for (const auto& rule: rules) {
            if (rule->execute(asrc, dst, hi)) {
                return -1;
            }
            if (dst.empty()) {
                return 0;
            }
            asrc = dst;
        }
        return 0;
    }

    Rules rules;
};


class RuleAlternation: public Rule {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo& hi) const override {
        dst.clear();
        for (const auto& rule: rules) {
            if (rule->execute(src, dst, hi)) {
                return -1;
            }
            if (!dst.empty()) {
                return 0;
            }
        }
        return 0;
    }
    Rules rules;
};


class RuleIdentity: public Rule {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo&) const override {
        dst = src;
        return 0;
    }
};


class RuleRestriction: public Rule {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo& hi) const override {
        dst.clear();
        if (!rule_.get()) {
            return 0;
        }
        if (!subject_.get() ||
            !subject_->match.get() ||
            !subject_->match->execute(hi.subject))
        {
            // we bypass subject restriction?
            return rule_->execute(src, dst, hi);
        }
        return 0;
    }

public:
    std::unique_ptr<Subject> subject_;
    std::unique_ptr<Rule> rule_;
};


class Rule_r: public Rule, public Pcre {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo&) const override {
        dst.clear();
        const char* mb = 0, *me = 0;
        std::array<int, 30> ovector;

        int count = exec(src, ovector);
        if (count == 0) {
            return 0;
        }

        dst.reserve(100);
        for (const char* ch = result_.c_str(); *ch; ++ch) {
            if (*ch == '$') {
                ch++;
                if (*ch && isdigit(*ch)) {
                    int ss = *ch - '0';
                    if (ss + 1 > count) {
                        return -1;
                    }

                    mb = src.data() + ovector[2 * ss];
                    me = src.data() + ovector[2 * ss + 1];
                    dst.append(mb, me);
                    continue;
                }
            }
            dst.append(1, *ch);
        }
        return 0;
    }

public:
    std::string result_;
};


class Rule_sr: public Rule, public Pcre {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo&) const override {
        dst.clear();
        size_t start = 0;
        std::array<int, 30> ovector;

        int count = exec(src, ovector);
        if (count == 0) {
            return 0;
        }

        start = includeMatch_ ? ovector[0] : ovector[1];

        if (start >= src.size()) {
            return -1;
        }

        std::string(src.begin() + start, src.end()).swap(dst);
        return 0;
    }

public:
    bool includeMatch_ = false;
};


class Rule_pr: public Rule, public Pcre {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo&) const override {
        dst.clear();
        std::array<int, 30> ovector;
        size_t end = 0;

        int count = exec(src, ovector);
        if (count == 0) {
            return 0;
        }

        end = includeMatch_ ? ovector[1] : ovector[0];

        if (end > src.size() || end <= 0) {
            return -1;
        }

        std::string(src.begin(), src.begin() + end).swap(dst);
        return 0;
    }

public:
    bool includeMatch_ = false;
};


class Rule_er: public Rule, public Pcre {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo&) const override {
        dst.clear();
        std::array<int, 30> ovector;

        std::vector<boost::string_ref> pieces;
        boost::string_ref in(src);

        do {
            int count = exec(in, ovector);
            if (count == 0) {
                break;
            }

            int start = ovector[0];
            int end = ovector[1];

            if (start < 0) {
                return -1;
            }

            if (start >= 0) {
                pieces.emplace_back(in.substr(0, start));
            }

            in.remove_prefix(end);

        } while (glob_ && !in.empty());

        if (pieces.empty()) {
            return 0;
        }

        for (const auto& piece: pieces) {
            dst.append(piece.data(), piece.size());
        }
        if (!in.empty()) {
            dst.append(in.data(), in.size());
        }

        return 0;
    }

public:
    bool glob_ = false;
};


class Rule_sl: public Rule {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo&) const override {
        dst.clear();
        if (lines_ <= 0 || src.empty()) {
            return -1;
        }

        size_t end = src.size(), start = end;

        int line = 0; // newlines skipped
        size_t pos = 0;
        while ((pos = src.find('\n', pos)) != std::string::npos) {
            ++pos; // skip '\n'
            ++line;
            if (line == lines_) {
                start = pos;
                break;
            }
        }

        if (start >= end || start == 0) {
            return -1;
        }

        std::string(src.begin() + start, src.begin() + end).swap(dst);
        return 0;
    }

public:
    int lines_ = 0;
};


class Rule_pl: public Rule {
public:
    int execute(const std::string& src, std::string& dst, const MessageHeaderInfo&) const override {
        dst.clear();
        if (lines_ <= 0 || src.empty()) {
            return -1;
        }

        size_t end = 0, start = end;

        int line = 0; // newlines skipped
        size_t pos = 0;
        while ((pos = src.find('\n', pos)) != std::string::npos) {
            ++pos; // skip '\n'
            ++line;
            if (line == lines_) {
                end = pos;
                break;
            }
        }

        if (end >= src.size()) {
            return -1;
        }

        std::string(src.begin() + start, src.begin() + end).swap(dst);
        return 0;
    }

public:
    int lines_ = 0;
};

}  // namespace NNotSoLiteSrv::NFirstline::NLib
