#include "subscription.h"

namespace NNotSoLiteSrv::NFirstline::NLib {

std::locale ruRU_utf8_loc;

std::unique_ptr<Subscription> CreateSubscription(
    const std::string& service,
    const std::string& subjstr,
    const SubscriptionSource* v)
{
    typedef std::multimap<std::string, SubscriptionHelper*>::const_iterator const_iterator;
    std::pair<const_iterator, const_iterator> p = v->smap.equal_range(service);
    for (const_iterator it = p.first; it != p.second; ++it) {
        SubscriptionHelper* s = it->second;
        Subject* subj = s->subject.get();
        bool s_match = (s->service.get() && *s->service == service);

        if (!s_match) {
            continue;
        }

        if (subj && subj->match.get()) {
            if (subj->match->execute(subjstr)) {
                continue;
            }
        }
        return std::make_unique<Subscription>(s, subjstr);
    }
    return {};
}


std::unique_ptr<Subscription> CreateSubscription(
    const std::string& from_email,
    const std::string& from_text,
    const std::string& subjstr,
    const SubscriptionSource* v)
{
    std::string a_from_email(from_email);
    std::transform(a_from_email.begin(), a_from_email.end(), a_from_email.begin(),
        [](char c) { return std::use_facet<std::ctype<char> >(ruRU_utf8_loc).tolower(c); });

    std::size_t at = std::string::npos;

    // First try to find a subscription that matches the whole of the originating email
    if (auto s = CreateSubscription_helper(a_from_email, from_text, subjstr, v)) {
        return s;
    } else if ((at = from_email.find('@')) != std::string::npos) {
        // If we failed, try to match a domain part of the originating email
        std::string a_from_email(from_email.begin() + at + 1, from_email.end());
        if (auto s = CreateSubscription_helper(a_from_email, from_text, subjstr, v)) {
            return s;
        }
    }
    // If all else failed, try fallback rules
    return CreateSubscription("fallback", subjstr, v);
}


std::unique_ptr<Subscription> CreateSubscription_helper(
    const std::string& from_email,
    const std::string& from_text,
    const std::string& subjstr,
    const SubscriptionSource* v)
{
    typedef std::multimap<std::string, SubscriptionHelper*>::const_iterator const_iterator;
    std::pair<const_iterator, const_iterator> p = v->smap.equal_range(from_email);
    for (const_iterator it=p.first; it!=p.second; ++it) {
        SubscriptionHelper* s = it->second;
        Subject* subj = s->subject.get();
        bool email_match = false;

        for (auto& from: s->froms) {
            if (from) { // must be true, just in case
                if (!from->match.get() || !from->match->execute(from_text)) {
                    email_match = true;
                    break;
                }
            }
        }
        if (!email_match) {
            continue;
        }

        if (subj && subj->match.get()) {
            if (subj->match->execute(subjstr))
                continue;
        }

        return std::make_unique<Subscription>(s, subjstr);
    }
    return {};
}

bool IsAuthorised(const Subscription* subscription, const char* authResults) {
    if (!subscription) {
        return false;
    }
    const SubscriptionHelper* ss = subscription->s;
    if (!ss->auth) { // is authorization required?
        return true;
    }
    // DKIM
    if (strstr(authResults, "dkim=pass") != nullptr) { // do not have to check for header.i here, since we already checked 'from' header field
        return true; // if DKIM passes, we are not interested in SPF
    } else if (strstr(authResults, "dkim=fail") != nullptr) {
        // if DKIM fails, we are not interested in SPF
        return false;
    } else if (strstr(authResults, "spf=pass") != nullptr) {
        // SPF
        bool emailMatch = false;
        const char smtpmail_literal[] = "smtp.mail=";
        if (const char* p = strstr(authResults, smtpmail_literal)) {
            p += (sizeof smtpmail_literal - 1);
            char smtpMail[256];
            memset(smtpMail, 0, sizeof smtpMail);
            strncpy(smtpMail, p, strcspn(p, "; "));

            for (const auto& fromPtr : ss->smtpfroms) {
                if (!fromPtr) {
                    continue;
                }
                const char* at = strchr(smtpMail, '@');
                if (strcasecmp(fromPtr->email.c_str(), smtpMail) == 0 ||
                    (at && strcasecmp(fromPtr->email.c_str(), (at + 1)) == 0))
                { // must be true, just in case
                    if (!fromPtr->match.get() || !fromPtr->match->execute(smtpMail)) {
                        emailMatch = true;
                        break;
                    }
                }
            }
        }
        return emailMatch;
    }
    return false;
}

} // namespace NNotSoLiteSrv::NFirstline::NLib
