#include "base_subject.h"

#include "refw_tokens.h"

#include <util/generic/hash_set.h>
#include <util/generic/iterator_range.h>

namespace NMthr {

namespace {

const THashMap<ui8, ui8> BRACKETS_MAP{
    {'(', ')'},
    {'[', ']'},
    {'{', '}'}
};

/**
 * @brief subj-fwd = subj-fwd-hdr subject subj-fwd-trl
 */
template <class Iterator>
TIteratorRange<Iterator> TrimSubjFwd(Iterator begin, Iterator end) {
    if (begin == end || *begin != '[' || *(end - 1) != ']') {
        return {begin, end};
    }

    auto innerBegin = LeftTrimWhiteSpaces(begin + 1, end);
    auto innerEnd = end - 1;

    auto iter = LeftTrimLongestToken(innerBegin, innerEnd, FW_TOKENS);
    if (iter != innerBegin && iter != innerEnd && *iter == ':') {
        return {++iter, innerEnd};
    }
    return {begin, end};
}

/**
 * brief subj-blob = "[" *BLOBCHAR "]"
 */
template <class Iterator>
Iterator LeftTrimSubjBlob(Iterator begin, Iterator end) {
    if (begin == end || *begin != '[') {
        return begin;
    }

    auto iter = std::find_if(begin + 1, end, [](auto ch) {
        return (ch == '[') || (ch == ']');
    });
    if (iter != end && *iter == ']') {
        return ++iter;
    }
    return begin;
}

/**
 * brief num-block = "(" *DIGIT ")" / "[" *DIGIT "]" / "{" *DIGIT "}"
 */
template <class Iterator>
Iterator LeftTrimNumberBlock(Iterator begin, Iterator end) {
    if (begin == end) {
        return begin;
    }
    auto bracket = BRACKETS_MAP.find(*begin);
    if (bracket == BRACKETS_MAP.end()) {
        return begin;
    }

    auto iter = std::find_if_not(begin + 1, end, [](auto ch) {
        return std::isdigit(ch);
    });

    if (iter != end && *iter == bracket->second) {
        return iter + 1;
    }
    return begin;
}

/**
 * @brief subj-trailer = "(" FW_TOKEN ")"
 */
template <class Iterator>
Iterator RightTrimSubjTrailer(Iterator begin, Iterator end) {
    if (begin == end || *(end - 1) != ')') {
        return end;
    }
    Iterator innerEnd = end - 1;
    auto iter = RightTrimLongestToken(begin, innerEnd, FW_TOKENS);
    if (iter != innerEnd && iter != begin && *(--iter) == '(') {
        return iter;
    }
    return end;
}

/**
 * @brief subj-trailer = "(" FW_TOKEN ")"
 */
template <class Iterator>
Iterator LeftTrimSubjTrailer(Iterator begin, Iterator end) {
    if (begin == end || *begin != '(') {
        return begin;
    }
    auto iter = LeftTrimLongestToken(begin + 1, end, FW_TOKENS);
    if (iter != end && *iter == ')') {
        return iter + 1;
    }
    return begin;
}

/**
 * @brief subj-blobs = *(*WSP subj-blob)
 */
template <class Iterator>
Iterator LeftTrimSubjBlobs(Iterator begin, Iterator end) {
    while (begin != end) {
        auto iter = LeftTrimWhiteSpaces(begin, end);
        iter = LeftTrimSubjBlob(iter, end);
        if (iter != begin) {
            begin = iter;
            continue;
        }
        break;
    }
    return begin;
}

/**
 * @brief subj-refwd = (RE_TOKEN [num-block] / FW_TOKEN) subj-blobs ":"
 */
template <class Iterator>
Iterator LeftTrimSubjReFwd(Iterator begin, Iterator end) {
    auto iter = LeftTrimLongestToken(begin, end, RE_TOKENS);
    if (iter != begin && iter != end) {
        iter = LeftTrimNumberBlock(iter, end);
    } else {
        iter = LeftTrimLongestToken(begin, end, FW_TOKENS);
    }

    if (iter == begin || iter == end) {
        return begin;
    }

    iter = LeftTrimSubjBlobs(iter, end);
    if (iter != begin && iter != end && *iter== ':') {
        return iter + 1;
    }
    return begin;
}

/**
 * @brief Removes specific subject artifacts from [begin, end).
 * @note Original algorithm is described in rfc5256.
 *      This differs in the following:
 *      * it handles additional tokens as well as "fwd" and "re";
 *      * it handles number block after "re" token;
 *      * it handles many subj-blobs instead of single in subj-refwd;
 *      * it does not handle subj-trailer within subj-refwd.
 * @see https://tools.ietf.org/rfc/rfc5256.txt "2.1 Base Subject".
 * @returns [begin, end) of "base subject".
 */
template <class Iterator>
TIteratorRange<Iterator> FindBaseSubject(Iterator begin, Iterator end) {
    while (begin != end) {
        end = RightTrimWhiteSpaces(begin, end);
        auto iter = RightTrimSubjTrailer(begin, end);
        if (iter != end) {
            end = iter;
            continue;
        }
        break;

    }

    while (begin != end) {
        begin = LeftTrimWhiteSpaces(begin, end);

        auto iter = LeftTrimSubjTrailer(begin, end);
        if (iter != begin) {
            begin = iter;
            continue;
        }

        auto range = TrimSubjFwd(begin, end);
        if (range.begin() != begin && range.end() != end) {
            begin = range.begin();
            end = range.end();
            continue;
        }

        iter = LeftTrimSubjBlob(begin, end);
        if (iter != begin) {
            begin = iter;
            continue;
        }

        iter = LeftTrimSubjReFwd(begin, end);
        if (iter != begin) {
            begin = iter;
            continue;
        }

        break;
    }

    return {begin, end};
}

} // namespace anonymous

TStringBuf FindBaseSubject(const TStringBuf& subject) {
    auto range = FindBaseSubject(subject.begin(), subject.end());
    return {range.begin(), range.end()};
}

} // namespace NMthr
