#include "merge_rules.h"
#include "util.h"

#include <mail/notsolitesrv/lib/mthr/src/reflection/trivial_subjects.h>

#include <util/generic/algorithm.h>
#include <util/charset/utf8.h>
#include <util/stream/file.h>
#include <util/system/fs.h>
#include <yamail/data/deserialization/yajl.h>

#include <algorithm>
#include <iterator>

namespace NMthr {

namespace {

template <typename T>
bool IsSetIntersectionEmpty(const TSet<T>& setA, const TSet<T>& setB) {
    auto itA = setA.cbegin();
    auto itB = setB.cbegin();
    while (itA != setA.end() && itB != setB.end()) {
        if (*itA == *itB) {
            return false;
        }
        if (*itA < *itB) {
            ++itA;
        } else {
            ++itB;
        }
    }
    return true;
}

} // namespace anonymous

TMergeRule MakeMergeRule(TOriginalMergeRule originalMergeRule) {
    return {
        .Id = originalMergeRule.id,
        .Priority = originalMergeRule.priority,
        .Field = std::move(originalMergeRule.field),
        .RotateDays = originalMergeRule.rotate_days.value_or(TMergeRule{}.RotateDays),
        .RotateMails = originalMergeRule.rotate_mails.value_or(TMergeRule{}.RotateMails),
        .MsgLabels = MakeSet<std::string>(std::move(originalMergeRule.message_labels)),
        .MsgTypes = MakeSet<NMail::EMessageType>(originalMergeRule.message_types),
        .Experiments = MakeSet<std::string>(std::move(originalMergeRule.experiments)),
        .IsTrivial = NMthr::MakeMaybe(originalMergeRule.triviality)
    };
}

TVector<TMergeRule> ReadMergeRulesFromJson(const std::string& json) {
    auto originalMergeRules{yamail::data::deserialization::fromJson<TOriginalMergeRules>(json)};
    TVector<TMergeRule> mergeRules;
    mergeRules.reserve(originalMergeRules.merge_rules.size());
    auto transformer = [](auto&& rule){return MakeMergeRule(std::move(rule));};
    std::transform(
        std::make_move_iterator(originalMergeRules.merge_rules.begin()),
        std::make_move_iterator(originalMergeRules.merge_rules.end()),
        std::back_inserter(mergeRules),
        std::move(transformer));
    return mergeRules;
}

TVector<TString> ReadTrivialSubjectsFromJson(const std::string& json) {
    auto originalTrivialSubjects{yamail::data::deserialization::fromJson<TOriginalTrivialSubjects>(json)};
    TVector<TString> trivialSubjects;
    trivialSubjects.reserve(originalTrivialSubjects.trivial_subjects.size());
    std::move(
        originalTrivialSubjects.trivial_subjects.begin(),
        originalTrivialSubjects.trivial_subjects.end(),
        std::back_inserter(trivialSubjects));
    return trivialSubjects;
}

TMergeRules::TMergeRules(
    const TVector<TMergeRule>& mergeRules,
    const TVector<TString>& trivialSubjects
)
    : Rules(mergeRules.begin(), mergeRules.end())
    , TrivialSubjects(trivialSubjects.begin(), trivialSubjects.end())
{
    Sort(Rules.begin(), Rules.end(), std::greater<>());
}

TMergeRules::TConstIterator TMergeRules::end() const {
    return Rules.end();
}

bool TMergeRules::IsSubjectTrivial(const TStringBuf& subject) const {
    if (!subject) {
        return true;
    }
    return TrivialSubjects.contains(ToLowerUTF8(subject));
}

bool TMergeRules::MatchBySubject(
    const TMergeRule& rule,
    const TStringBuf& subject) const
{
    if (!rule.IsTrivial.Defined()) {
        return true;
    }
    return rule.IsTrivial == IsSubjectTrivial(subject);
}

bool TMergeRules::MatchByMsgLabel(
    const TMergeRule& rule,
    const std::string& msgLabel)
{
    if (!rule.MsgLabels) {
        return true;
    }
    return rule.MsgLabels.contains(msgLabel);
}

bool TMergeRules::MatchByMsgTypes(
    const TMergeRule& rule,
    const TSet<NMail::EMessageType>& msgTypes)
{
    if (!rule.MsgTypes) {
        return true;
    }
    auto notInMsgTypes = [&msgTypes](const auto& msgType) {
        return !msgTypes.contains(msgType);
    };
    return CountIf(rule.MsgTypes.begin(), rule.MsgTypes.end(), notInMsgTypes) == 0;
}

bool TMergeRules::MatchByExperiments(
    const TMergeRule& rule,
    const TSet<std::string>& experiments)
{
    if (!rule.Experiments) {
        return true;
    }
    return !IsSetIntersectionEmpty(rule.Experiments, experiments);
}

TMergeRules::TConstIterator TMergeRules::find(
    const TStringBuf& baseSubject,
    const std::string& label,
    const TSet<NMail::EMessageType>& msgTypes,
    const TSet<std::string>& experiments) const
{
    for (auto iter = Rules.begin(); iter != Rules.end(); ++iter) {
        if (!MatchBySubject(*iter, baseSubject)) {
            continue;
        }
        if (!MatchByMsgLabel(*iter, label)) {
            continue;
        }
        if (!MatchByMsgTypes(*iter, msgTypes)) {
            continue;
        }
        if (!MatchByExperiments(*iter, experiments)) {
            continue;
        }
        return iter;
    }
    return Rules.end();
}

} // namespace NMthr
