#include "message_id.h"

#include <boost/spirit/include/qi.hpp>
#include <util/generic/algorithm.h>

namespace NMthr {

namespace NQi = boost::spirit::qi;

template <class Iterator>
struct TMessageIdGrammar : public NQi::grammar<Iterator, TString()> {
    TMessageIdGrammar() : TMessageIdGrammar::base_type(MessageId) {
        using NQi::alnum;
        using NQi::blank;
        using NQi::space;
        using NQi::char_;
        using NQi::raw;
        using NQi::omit;

        AlphaNumText = alnum | char_("!#$%&'*+/=?^_`{|}~-");
        DotAtomText = +AlphaNumText >> *('.' >> +AlphaNumText);

        QuotedContent = (char_ - char_("\\\"")) | ('\\' >> char_);
        QuotedString = '"' >> *(-blank >> QuotedContent) >> -blank >> '"';

        SubDomain = alnum >> *(alnum | '-');
        Domain = SubDomain % '.';

        Mailbox = (DotAtomText | QuotedString) >> '@'
            >> (Domain | ('[' >> DotAtomText >> ']'));

        MessageId = raw['<' >> DotAtomText >> '@' >> Domain >> '>'];
    }

    NQi::rule<Iterator> AlphaNumText;
    NQi::rule<Iterator> DotAtomText;
    NQi::rule<Iterator> QuotedContent;
    NQi::rule<Iterator> QuotedString;
    NQi::rule<Iterator> SubDomain;
    NQi::rule<Iterator> Domain;
    NQi::rule<Iterator> Mailbox;
    NQi::rule<Iterator, TString()> MessageId;
};

bool IsValidMessageId(const TString& messageId) {
    TMessageIdGrammar<TString::const_iterator> grammar;
    return NQi::parse(messageId.begin(), messageId.end(), grammar);
}

} // namespace NMthr
