#include <mail/notsolitesrv/lib/mthr/src/exception.h>
#include <mail/notsolitesrv/lib/mthr/src/mail_threads.h>
#include <mail/notsolitesrv/lib/mthr/src/fnv_hash.h>

#include <mail/message_types/lib/message_types.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/gtest.h>

#include <string>
#include <vector>

using namespace NMthr;

TEST(CalcHashFromLowerUTF8, TestBasic) {
    struct Fixture {
        TString msgId;
        ui64 hash;
    };
    std::vector<Fixture> fixtures{
        {"<76989358.14.1511266366225.JavaMail.aero@aero-agent-e-185>", 3877811958869977621ULL},
        {"congratulations", 3059512114109226478ULL},
        {"вакансия", 9350279830116798128ULL},
        {"message9N3cDAzChkcWpgGlk", 14294244891726659881ULL}
    };
    for (const auto& fixture : fixtures) {
        EXPECT_EQ(CalcHashFromLowerUTF8(fixture.msgId), fixture.hash);
    }
}

TEST(CalcMailThreadInfo, TestEmpty) {
    TMergeRules mergeRules({}, {});
    EXPECT_THROW(FindMergeRuleSafe({}, mergeRules), TMthrException);
}

TEST(CalcMailThreadInfo, TestTrivialSubject) {
    TMessageInfo msgInfo{
        .HeaderFromDomain = "mail.yandex-team.ru",
        .Subject = "Re: documents",
        .MessageId = "<1107794284.159.1511266400874.JavaMail.aero@aero-agent-e-35.haze.yandex.net>",
        .References = {"<hd2GOwQ3mHDs5CC@hst3.ru>"},
    };

    const std::string mergeRulesJson{R"({
        "merge_rules": [
            {
                "id": 2,
                "priority": 0,
                "field": "references",
                "triviality": true
            }
        ]
    })"};

    auto mergeRules = ReadMergeRulesFromJson(mergeRulesJson);
    auto rule = FindMergeRuleSafe(msgInfo, {mergeRules, {"documents"}});
    auto threadInfo = CalcMailThreadInfo(msgInfo, rule);
    EXPECT_EQ(rule.Id, 2);

    EXPECT_TRUE(threadInfo.MergePolicy == EMergePolicy::References);

    const auto& limits = threadInfo.Limits;
    EXPECT_EQ(limits.DaysLimit, 90);
    EXPECT_EQ(limits.CountLimit, 1000);

    const auto& hash = threadInfo.Hash;
    EXPECT_TRUE(hash.NameSpace == EHashNameSpace::Subject);
    EXPECT_EQ(hash.Value, 6143164950382953529ULL);

    EXPECT_EQ(threadInfo.MessageIdHash, 14722625308284324442ULL);
    EXPECT_EQ(threadInfo.InReplyToHash, 0);
    TVector<TString> expectedMessageIds({
        "<1107794284.159.1511266400874.JavaMail.aero@aero-agent-e-35.haze.yandex.net>",
        "<hd2GOwQ3mHDs5CC@hst3.ru>"
    });
    EXPECT_EQ(threadInfo.MessageIds, expectedMessageIds);
}

TEST(CalcMailThreadInfo, TestForceNewThread) {
    using namespace NMail;

    TMessageInfo msgInfo{
        .HeaderFromDomain = "v138mth6BYxBYZEz.ru",
        .Subject = "messagev138mth6BYxBYZEz",
        .MessageId = "<955362309.191.1511266399156.JavaMail.aero@aero-agent-e-185>",
        .References = {"<v138mth6BYxBYZEz@lm.ru>"},
        .MessageTypes = {MT_BOUNCE, MT_LIVE_MAIL, MT_S_TRAVEL}
    };

    const std::string mergeRulesJson{R"({
        "merge_rules": [
            {
                "id": 10,
                "priority": 4,
                "field": "force_new",
                "message_types": [8]
            }
        ]
    })"};

    auto mergeRules = ReadMergeRulesFromJson(mergeRulesJson);
    auto rule = FindMergeRuleSafe(msgInfo, {mergeRules, {}});
    auto threadInfo = CalcMailThreadInfo(msgInfo, rule);

    EXPECT_TRUE(threadInfo.MergePolicy == EMergePolicy::ForceNew);

    const auto& limits = threadInfo.Limits;
    EXPECT_EQ(limits.DaysLimit, 90);
    EXPECT_EQ(limits.CountLimit, 1000);

    const auto& hash = threadInfo.Hash;
    EXPECT_TRUE(hash.NameSpace == EHashNameSpace::Subject);
    EXPECT_EQ(hash.Value, 277028877244586148ULL);

    EXPECT_EQ(threadInfo.ReferenceHashes, TVector<ui64>({13823907676205527079ULL, 14201686274467093711ULL}));
    EXPECT_EQ(threadInfo.MessageIdHash, 13823907676205527079ULL);
    EXPECT_EQ(threadInfo.InReplyToHash, 0);
    TVector<TString> expectedMessageIds({
        "<955362309.191.1511266399156.JavaMail.aero@aero-agent-e-185>",
        "<v138mth6BYxBYZEz@lm.ru>"
    });
    EXPECT_EQ(threadInfo.MessageIds, expectedMessageIds);
}

TEST(CalcMailThreadInfo, TestHash) {
    TMessageInfo msgInfo{
        .HeaderFromDomain = "mail.yandex-team.ru",
        .Subject = "答复: message答复! N3cDAzChkcWpgGlk",
        .MessageId = "<1620921199.74.1511266366597.JavaMail.aero@aero-agent-f-30.haze.yandex.net>",
        .References = {"<sameref@ru>"},
    };

    const std::string mergeRulesJson{R"({
        "merge_rules": [
            {
                "id": 1,
                "priority": 0,
                "field": "base_subject",
                "triviality": false,
                "rotate_days": 90
            }
        ]
    })"};

    auto mergeRules = ReadMergeRulesFromJson(mergeRulesJson);
    auto rule = FindMergeRuleSafe(msgInfo, {mergeRules, {}});
    auto threadInfo = CalcMailThreadInfo(msgInfo, rule);

    EXPECT_TRUE(threadInfo.MergePolicy == EMergePolicy::Hash);

    const auto& limits = threadInfo.Limits;
    EXPECT_EQ(limits.DaysLimit, 90);
    EXPECT_EQ(limits.CountLimit, 1000);

    const auto& hash = threadInfo.Hash;
    EXPECT_TRUE(hash.NameSpace == EHashNameSpace::Subject);
    EXPECT_EQ(hash.Value, 17853029033725563854ULL);

    EXPECT_EQ(threadInfo.ReferenceHashes, TVector<ui64>({311545520275728254ULL, 6464062408867917559ULL}));
    EXPECT_EQ(threadInfo.MessageIdHash, 311545520275728254ULL);
    EXPECT_EQ(threadInfo.InReplyToHash, 0);
    TVector<TString> expectedMessageIds({
        "<1620921199.74.1511266366597.JavaMail.aero@aero-agent-f-30.haze.yandex.net>",
        "<sameref@ru>"
    });
    EXPECT_EQ(threadInfo.MessageIds, expectedMessageIds);
}

TEST(CalcMailThreadInfo, TestMessageLables) {
    TMessageInfo msgInfo{
        .HeaderFromDomain = "N3cDAzChkcWpgGlk.ua",
        .Subject = "!",
        .MessageId = "<1326120812.122.1511266373696.JavaMail.aero@aero-agent-f-30.haze.yandex.net>",
        .References = {"<N3cDAzChkcWpgGlk@refffff>"},
        .DomainLabel = "vtnrf0jira"
    };

    const std::string mergeRulesJson{R"({
        "merge_rules": [
            {
                "id": 3,
                "priority": 5,
                "field": "references",
                "message_labels": [
                    "jira",
                    "vtnrf0jira",
                    "startrack",
                    "livejournal",
                    "vtnrf0livejournal",
                    "yaru",
                    "vtnrf0yaru",
                    "yasupport",
                    "powny"
                ]
            }
        ]
    })"};

    auto mergeRules = ReadMergeRulesFromJson(mergeRulesJson);
    auto rule = FindMergeRuleSafe(msgInfo, {mergeRules, {}});
    auto threadInfo = CalcMailThreadInfo(msgInfo, rule);

    EXPECT_TRUE(threadInfo.MergePolicy == EMergePolicy::References);

    const auto& limits = threadInfo.Limits;
    EXPECT_EQ(limits.DaysLimit, 90);
    EXPECT_EQ(limits.CountLimit, 1000);

    const auto& hash = threadInfo.Hash;
    EXPECT_TRUE(hash.NameSpace == EHashNameSpace::Subject);
    EXPECT_EQ(hash.Value, 12638153115695167486ULL);

    EXPECT_EQ(threadInfo.ReferenceHashes, TVector<ui64>({15505377509885665098ULL, 11213757536630298060ULL}));
    EXPECT_EQ(threadInfo.MessageIdHash, 15505377509885665098ULL);
    EXPECT_EQ(threadInfo.InReplyToHash, 0);
    TVector<TString> expectedMessageIds({
        "<1326120812.122.1511266373696.JavaMail.aero@aero-agent-f-30.haze.yandex.net>",
        "<N3cDAzChkcWpgGlk@refffff>"
    });
    EXPECT_EQ(threadInfo.MessageIds, expectedMessageIds);
}
