#include <mail/notsolitesrv/lib/mthr/src/merge_rules.h>

#include <mail/notsolitesrv/lib/mthr/ut/util/merge_rules.h>

#include <library/cpp/testing/unittest/gtest.h>

#include <string>
#include <utility>

using NMail::EMessageType;
using NMail::MT_DELIVERY;
using NMail::MT_REGISTRATION;
using NMthr::MakeMergeRule;
using NMthr::ReadMergeRulesFromJson;
using NMthr::ReadTrivialSubjectsFromJson;
using NMthr::TMergeRule;
using NMthr::TMergeRules;
using NMthr::TOriginalMergeRule;

TEST(MergeRules, MakeMergeRule) {
    TOriginalMergeRule originalRule;
    originalRule.id = 0;
    originalRule.priority = 1;
    originalRule.field = "field0";

    TMergeRule expectedRule;
    expectedRule.Id = 0;
    expectedRule.Priority = 1;
    expectedRule.Field = "field0";

    EXPECT_TRUE(expectedRule == MakeMergeRule(originalRule));

    originalRule.id = 1;
    originalRule.priority = 2;
    originalRule.field = "field1";
    originalRule.rotate_days = 3;
    originalRule.rotate_mails = 4;
    originalRule.message_labels = {{"label0", "label1"}};
    originalRule.message_types = {{1, 2}};
    originalRule.experiments = {{"experiment0", "experiment1"}};
    originalRule.triviality = true;

    expectedRule.Id = 1;
    expectedRule.Priority = 2;
    expectedRule.Field = "field1";
    expectedRule.RotateDays = 3;
    expectedRule.RotateMails = 4;
    expectedRule.MsgLabels = {"label0", "label1"};
    expectedRule.MsgTypes = {MT_DELIVERY, MT_REGISTRATION};
    expectedRule.Experiments = {"experiment0", "experiment1"};
    expectedRule.IsTrivial = true;

    EXPECT_TRUE(expectedRule == MakeMergeRule(originalRule));
}

TEST(MergeRules, ReadMergeRulesFromJson) {
    const std::string mergeRulesJson{R"({
        "merge_rules": [
            {
                "id": 0,
                "priority": 1,
                "field": "field0"
            }, {
                "id": 1,
                "priority": 2,
                "field": "field1",
                "rotate_days": 3,
                "rotate_mails": 4,
                "message_labels": ["label0", "label1"],
                "message_types": [1, 2],
                "experiments": ["experiment0", "experiment1"],
                "triviality": true
            }
        ]
    })"};
    const auto mergeRules{ReadMergeRulesFromJson(mergeRulesJson)};
    EXPECT_EQ(2, mergeRules.size());

    TMergeRule defaultMergeRule;
    EXPECT_EQ(0, mergeRules[0].Id);
    EXPECT_EQ(1, mergeRules[0].Priority);
    EXPECT_EQ("field0", mergeRules[0].Field);
    EXPECT_EQ(defaultMergeRule.RotateDays, mergeRules[0].RotateDays);
    EXPECT_EQ(defaultMergeRule.RotateMails, mergeRules[0].RotateMails);
    EXPECT_EQ(defaultMergeRule.MsgLabels, mergeRules[0].MsgLabels);
    EXPECT_EQ(defaultMergeRule.MsgTypes, mergeRules[0].MsgTypes);
    EXPECT_EQ(defaultMergeRule.Experiments, mergeRules[0].Experiments);
    EXPECT_EQ(defaultMergeRule.IsTrivial, mergeRules[0].IsTrivial);

    EXPECT_EQ(1, mergeRules[1].Id);
    EXPECT_EQ(2, mergeRules[1].Priority);
    EXPECT_EQ("field1", mergeRules[1].Field);
    EXPECT_EQ(3, mergeRules[1].RotateDays);
    EXPECT_EQ(4, mergeRules[1].RotateMails);
    EXPECT_EQ((TSet<std::string>{"label0", "label1"}), mergeRules[1].MsgLabels);
    EXPECT_EQ((TSet<EMessageType>{MT_DELIVERY, MT_REGISTRATION}), mergeRules[1].MsgTypes);
    EXPECT_EQ((TSet<std::string>{"experiment0", "experiment1"}), mergeRules[1].Experiments);
    EXPECT_EQ(true, mergeRules[1].IsTrivial);
}

TEST(MergeRules, EmptyRulesNothingFound) {
    TMergeRules mergeRules;
    auto iter = mergeRules.find("", "", {}, {});
    EXPECT_TRUE(iter == mergeRules.end());
}

TEST(MergeRules, FindTrivialByEmptySubject) {
    TMergeRule mergeRule{.Id = 1, .Priority = 0, .Field = "base_subject", .IsTrivial = true};
    TMergeRules mergeRules{{std::move(mergeRule)}, {}};
    auto iter = mergeRules.find("", "", {}, {});
    EXPECT_TRUE(iter != mergeRules.end());
    EXPECT_EQ(iter->Id, 1);
}

TEST(MergeRules, FindTrivialByNonEmptySubject) {
    TMergeRule mergeRule{.Id = 1, .Priority = 0, .Field = "base_subject", .IsTrivial = true};
    TMergeRules mergeRules{{std::move(mergeRule)}, {"subject1"}};
    auto iter = mergeRules.find("Subject1", "", {}, {});
    EXPECT_TRUE(iter != mergeRules.end());
    EXPECT_EQ(iter->Id, 1);
    iter = mergeRules.find("Subject2", "", {}, {});
    EXPECT_TRUE(iter == mergeRules.end());
}

TEST(MergeRules, FindNonTrivialByNonEmptySubject) {
    TMergeRule mergeRule{.Id = 1, .Priority = 0, .Field = "base_subject", .IsTrivial = false};
    TMergeRules mergeRules{{std::move(mergeRule)}, {}};
    auto iter = mergeRules.find("foo", "", {}, {});
    EXPECT_TRUE(iter != mergeRules.end());
    EXPECT_EQ(iter->Id, 1);
    iter = mergeRules.find("bar", "", {}, {});
    EXPECT_TRUE(iter != mergeRules.end());
    EXPECT_EQ(iter->Id, 1);
    iter = mergeRules.find("", "", {}, {});
    EXPECT_TRUE(iter == mergeRules.end());
}

TEST(MergeRules, FindByMsgLabel) {
    const std::string label{"startrack"};
    TMergeRule mergeRule{.Id = 1, .Priority = 5, .Field = "references", .MsgLabels = {label}};
    TMergeRules mergeRules{{std::move(mergeRule)}, {}};
    auto iter = mergeRules.find("", label, {}, {});
    EXPECT_TRUE(iter != mergeRules.end());
    EXPECT_EQ(iter->Id, 1);
    iter = mergeRules.find("", "foo", {}, {});
    EXPECT_TRUE(iter == mergeRules.end());
}

TEST(MergeRules, FindByMsgTypes) {
    TMergeRule mergeRule{.Id = 1, .Priority = 5, .Field = "force_new",
        .MsgTypes = {MT_DELIVERY, MT_REGISTRATION}};
    TMergeRules mergeRules{{std::move(mergeRule)}, {}};
    auto iter = mergeRules.find("", "", {MT_DELIVERY, MT_REGISTRATION}, {});
    EXPECT_TRUE(iter != mergeRules.end());
    EXPECT_EQ(iter->Id, 1);
    iter = mergeRules.find("", "", {MT_DELIVERY}, {});
    EXPECT_TRUE(iter == mergeRules.end());
}

TEST(MergeRules, FindByExperiments) {
    TMergeRule mergeRule{.Id = 1, .Priority = 5, .Field = "force_new", .Experiments = {"A", "B"}};
    TMergeRules mergeRules{{std::move(mergeRule)}, {}};
    auto iter = mergeRules.find("", "", {}, {"A"});
    EXPECT_TRUE(iter != mergeRules.end());
    EXPECT_EQ(iter->Id, 1);
    iter = mergeRules.find("", "", {}, {"C", "D"});
    EXPECT_TRUE(iter == mergeRules.end());
    iter = mergeRules.find("", "", {}, {"C", "B"});
    EXPECT_TRUE(iter != mergeRules.end());
    EXPECT_EQ(iter->Id, 1);
}

TEST(MergeRules, FindByExperimentsNoRulesWithExperiments) {
    const std::string label{"startrack"};
    TMergeRule mergeRule{.Id = 1, .Priority = 5, .Field = "force_new", .MsgLabels = {label}};
    TMergeRules mergeRules{{std::move(mergeRule)}, {}};
    const std::string experiment{"A"};
    auto iter = mergeRules.find("", "", {}, {experiment});
    EXPECT_TRUE(iter == mergeRules.end());
    iter = mergeRules.find("", label, {}, {experiment});
    EXPECT_TRUE(iter != mergeRules.end());
    EXPECT_EQ(iter->Id, 1);
}

TEST(MergeRules, ReadTrivialSubjectsFromJson) {
    const std::string trivialSubjectsJson{R"({"trivial_subjects": ["subject0", "subject1", "subject2"]})"};
    const auto trivialSubjects{ReadTrivialSubjectsFromJson(trivialSubjectsJson)};
    EXPECT_EQ(3, trivialSubjects.size());
    EXPECT_EQ("subject0", trivialSubjects[0]);
    EXPECT_EQ("subject1", trivialSubjects[1]);
    EXPECT_EQ("subject2", trivialSubjects[2]);
}
