#include "client.h"
#include "client_helpers.h"
#include <mail/notsolitesrv/src/errors.h>
#include <mail/notsolitesrv/src/config/httpcall.h>
#include <mail/notsolitesrv/src/http/client.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/notsolitesrv/src/util/string.h>
#include <yplatform/coroutine.h>
#include <yplatform/log.h>

#include <util/generic/algorithm.h>
#include <util/generic/yexception.h>

namespace NNotSoLiteSrv::NBlackbox {

const std::string BB {"BB"};

namespace NDetail {

const std::string DBFIELDS{
    "subscription.login.-,"
    "subscription.login_rule.-,"
    "subscription.suid.-,"
    "account_info.fio.uid,"
    "account_info.country.uid,"
    "account_info.lang.uid,"
    "account_info.reg_date.uid,"
    "userphones.confirmed.uid"
};
const std::string DBFIELDS_MAILISH{
    "account_info.fio.uid,"
    "account_info.country.uid,"
    "account_info.lang.uid,"
    "account_info.reg_date.uid,"
    "userphones.confirmed.uid"
};
const std::string ATTRIBUTES{
    "1031"
};

#include <yplatform/yield.h>

class TResolver {
public:
    using TYieldCtx = yplatform::yield_context<TResolver>;

    TResolver(
        TContextPtr ctx,
        NHttp::TClientPtr httpClient,
        const std::string& email,
        bool getByUid,
        const TUsers& users,
        TCallback cb
    )
        : Ctx(ctx)
        , HttpClient(httpClient)
        , Email(email)
        , GetByUid(getByUid)
        , Users(users)
        , Callback(std::move(cb))
    {
        Y_ENSURE(GetByUid || Users.size() == 1, "could not load more than one user from BB by login");
    }

    void operator()(
        TYieldCtx yctx,
        TErrorCode ec = TErrorCode(),
        const NHttp::TResponse& result = NHttp::TResponse())
    {
        try {
            reenter (yctx) {
                if (!GetByUid && Users.size() != 1) {
                    NSLS_LOG_CTX_ERROR(logdog::message="unable to resolve many users by login", logdog::where_name=BB);
                    ec = EError::DeliveryInternal;
                    yield break;
                }
                if (GetByUid && !AllOf(Users, [](const NUser::TUser& u) { return !u.Uid.empty(); })) {
                    NSLS_LOG_CTX_ERROR(logdog::message="bb_client error [" + LoginData() + "]: empty mailish user uid", logdog::where_name=BB);
                    ec = EError::UserInvalid;
                    yield break;
                }

                yield HttpClient->Get(MakeUrl(), yctx);

                if (ec) {
                    NSLS_LOG_CTX_ERROR(
                        logdog::message="bb_client error [" + LoginData() + "]: " + ec.message(),
                        logdog::where_name=BB);
                    yield break;
                }

                if (result.status / 100 != 2) {
                    NSLS_LOG_CTX_ERROR(
                        logdog::message=
                            "blackbox returns [" + LoginData() + "] " +
                            "(" + std::to_string(result.status) + "): " + result.reason,
                        logdog::where_name=BB);
                    ec = EError::DeliveryInternal;
                    yield break;
                }
                ec = ParseUserInfo(result);

                yield break;
            }
        } catch (const std::exception& e) {
            NSLS_LOG_CTX_ERROR(
                logdog::message="bb_client exception for [" + LoginData() + "]",
                logdog::exception=e,
                logdog::where_name=BB);
            ec = EError::DeliveryInternal;
            for (NUser::TUser& u: Users) {
                u.DeliveryResult.ErrorCode = ec;
            }
            return Callback(ec);
        }

        if (yctx.is_complete()) {
            if (ec) {
                for (NUser::TUser& u: Users) {
                    u.DeliveryResult.ErrorCode = ec;
                }
            }
            Callback(ec);
        }
    }

private:
    std::string LoginData() const {
        std::string ret;
        if (GetByUid) {
            ret.append("mailish(").append(NUtil::JoinBy(Users, ",", std::mem_fn(&NUser::TUser::Uid))).append(")");
        } else {
            ret = Email;
        }

        return ret;
    }

    TErrorCode ParseUserInfo(const NHttp::TResponse& result) {
        TErrorCode ec;
        for (NUser::TUser& u: Users) {
            u.Status = NUser::ELoadStatus::Loaded;
        }

        try {
            auto usersData = GetBlackboxData(result.body);
            ec = FillUsersWithBlackboxData(usersData, GetByUid, Users);
        } catch (const std::exception& e) {
            NSLS_LOG_CTX_ERROR(
                logdog::message="bb_client exception for [" + LoginData() + "]",
                logdog::exception=e,
                logdog::where_name=BB);
            ec = EError::UserInvalid;
            for (NUser::TUser& u: Users) {
                u.DeliveryResult.ErrorCode = EError::UserInvalid;
            }
        }
        return ec;
    }

    std::string MakeUrl() const {
        std::string params;
        if (GetByUid) {
            params = NHttp::UrlEncode({
                {"method",     "userinfo"},
                {"uid",        NUtil::JoinBy(Users, ",", std::mem_fn(&NUser::TUser::Uid))},
                {"userip",     "127.0.0.1"},
                {"emails",     "getdefault"},
                {"dbfields",   DBFIELDS_MAILISH},
                {"attributes", ATTRIBUTES},
                {"format",     "json"}
            });
        } else {
            std::string login;
            NUtil::ExtractLogin(Email, login);

            params = NHttp::UrlEncode({
                {"method",     "userinfo"},
                {"login",      login},
                {"sid",        "smtp"},
                {"userip",     "127.0.0.1"},
                {"emails",     "getdefault"},
                {"dbfields",   DBFIELDS},
                {"attributes", ATTRIBUTES},
                {"format",     "json"}
            });
        }
        std::string url = Ctx->GetConfig()->Blackbox->Url;
        NHttp::AppendUriParams(url, params);
        return url;
    }

    TContextPtr Ctx;
    NHttp::TClientPtr HttpClient;
    const std::string& Email;
    bool GetByUid;
    TUsers Users;
    TCallback Callback;
};

#include <yplatform/unyield.h>

} // namespace NDetail

void GetUser(
    TContextPtr ctx,
    const std::string& email,
    bool getByUid,
    NUser::TUser& userInfo,
    TCallback cb)
{
    return GetUserWithHttpClient(
        ctx,
        email,
        getByUid,
        userInfo,
        NHttp::CreateClientByName(ctx, BB, *ctx->GetConfig()->Blackbox, "ghttp"),
        std::move(cb));
}

void GetUserWithHttpClient(
    TContextPtr ctx,
    const std::string& email,
    bool getByUid,
    NUser::TUser& userInfo,
    NHttp::TClientPtr httpClient,
    TCallback cb)
{
    auto resolver = std::make_shared<NDetail::TResolver>(ctx, httpClient, email, getByUid, TUsers{userInfo}, std::move(cb));
    yplatform::spawn(resolver);
}

void GetUsers(
    TContextPtr ctx,
    TUsers&& usersInfo,
    TCallback cb)
{
    return GetUsersWithHttpClient(
        ctx,
        std::move(usersInfo),
        NHttp::CreateClientByName(ctx, BB, *ctx->GetConfig()->Blackbox, "ghttp"),
        std::move(cb));
}

void GetUsersWithHttpClient(
    TContextPtr ctx,
    TUsers&& usersInfo,
    NHttp::TClientPtr httpClient,
    TCallback cb)
{
    auto resolver = std::make_shared<NDetail::TResolver>(ctx, httpClient, "", true, usersInfo, std::move(cb));
    yplatform::spawn(resolver);
}

} // namespace NNotSoLiteSrv::NBlackbox
