#include "client_helpers.h"

#include <mail/notsolitesrv/src/util/optional.h>

#include <util/generic/algorithm.h>
#include <util/generic/yexception.h>

#include <boost/property_tree/json_parser.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>

namespace NNotSoLiteSrv::NBlackbox {

using NUtil::ConvertOptional;

void CheckUserData(NUser::TUser& user, bool getByUid) {
    if (user.Uid.empty()) {
        user.DeliveryResult.ErrorCode = EError::UserNotFound;
    }

    if (!getByUid) {
        if (user.LoginRule == 0 || user.Suid.empty()) {
            user.DeliveryResult.ErrorCode = EError::UserInvalid;
        }
    }
}

std::string GetDefaultEmail(boost::optional<const boost::property_tree::ptree&>&& addressList) {
    if (!addressList || addressList->empty()) {
        return {};
    }

    for (const auto& address: *addressList) {
        if (address.second.get("default", false)) {
            return address.second.get("address", "");
        }
    }

    return {};
}

boost::property_tree::ptree GetBlackboxData(const std::string& data) {
    std::istringstream is(data);
    boost::property_tree::ptree pt;
    boost::property_tree::read_json(is, pt);

    return pt.get_child("users");
}

TErrorCode FillUsersWithBlackboxData(
    const boost::property_tree::ptree& usersData,
    bool getByUid,
    TUsers& users)
{
    Y_ENSURE(getByUid || usersData.size() == 1, "BB returns zero or many users for request by login");
    TErrorCode ec;

    for (const auto& [key, userData]: usersData) {
        auto id = userData.get<std::string>("id");

        boost::optional<NUser::TUser&> user;
        if (!getByUid) {
            user.emplace(users.front().get());
        }
        if (id.empty() || id == "0") {
            if (user) {
                user->DeliveryResult.ErrorCode = EError::UserNotFound;
            }
            ec = EError::UserNotFound;
            break;
        }

        if (getByUid) {
            if (auto userIt = FindIf(users, [id](const NUser::TUser& u){ return u.Uid == id; }); userIt != users.end()) {
                user.emplace(userIt->get());
            } else {
                continue;
            }
        }

        if (!user) {
            continue;
        }
        try {
            FillUserWithBlackboxData(userData, getByUid, *user);
        } catch (const std::exception& e) {
            user->DeliveryResult.ErrorCode = EError::UserInvalid;
        }
    }

    return ec;
}

void FillUserWithBlackboxData(const boost::property_tree::ptree& userData, bool getByUid, NUser::TUser& user) {
    using TPath = boost::property_tree::path;
    try {
        user.Uid = userData.get<std::string>("uid.value");
    } catch (const boost::property_tree::ptree_bad_path& e) {
        user.DeliveryResult.ErrorCode = EError::UserNotFound;
        return;
    }

    user.OrgId = ConvertOptional(userData.get_optional<TOrgId>("attributes.1031"));
    user.Karma = userData.get<int>("karma.value");
    user.KarmaStatus = userData.get<int>("karma_status.value");
    if (!getByUid) {
        user.LoginRule = userData.get<int>(TPath("dbfields/subscription.login_rule.-", '/'));
        user.Suid = userData.get(TPath("dbfields/subscription.suid.-", '/'), "");
    }
    auto regDateString = userData.get<std::string>(TPath("dbfields/account_info.reg_date.uid", '/'));
    if (!regDateString.empty()) {
        user.RegistrationDate = ParseDateTime(regDateString);
    }
    user.Country = userData.get(TPath("dbfields/account_info.country.uid", '/'), "");
    user.DefaultEmail = GetDefaultEmail(userData.get_child_optional("address-list"));
    user.FIO = userData.get(TPath("dbfields/account_info.fio.uid", '/'), "");

    user.Status = NUser::ELoadStatus::Found;
    CheckUserData(user, getByUid);
}

time_t ParseDateTime(const std::string& data) {
    namespace PTime = boost::posix_time;
    return PTime::to_time_t(PTime::time_from_string(data));
}

} // namespace NNotSoLiteSrv::NBlackbox
