#include "client.h"
#include <mail/notsolitesrv/src/config/httpcall.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/library/utf8/utf8.h>

#include <butil/email/helpers.h>

#include <yplatform/coroutine.h>
#include <yplatform/log.h>

#include <boost/property_tree/json_parser.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>

namespace NNotSoLiteSrv::NBlackWhiteList {

const std::string BWLIST {"BWLIST"};

namespace NDetail {

#include <yplatform/yield.h>

class TRequester {
public:
    using TYieldCtx = yplatform::yield_context<TRequester>;

    TRequester(
        TContextPtr ctx,
        NHttp::TClientPtr httpClient,
        const std::string& uid,
        TCallback cb
    )
        : Ctx(ctx)
        , HttpClient(httpClient)
        , Uid(uid)
        , Callback(std::move(cb))
        , BWList(std::make_shared<TList>())
    {
    }

    void operator()(
        TYieldCtx yctx,
        TErrorCode ec = TErrorCode(),
        const NHttp::TResponse& result = NHttp::TResponse())
    {
        try {
            reenter (yctx) {
                yield {
                    std::string url = Ctx->GetConfig()->BWList->Url;
                    NHttp::AppendUriParams(url, NHttp::UrlEncode({{"uid", Uid}}));

                    return HttpClient->Get(url, yctx);
                }

                if (ec) {
                    NSLS_LOG_CTX_ERROR(
                        log::uid=Uid,
                        logdog::message="bwlist_client error: " + ec.message(),
                        logdog::where_name=BWLIST);
                    yield break;
                }

                if (result.status / 100 != 2) {
                    NSLS_LOG_CTX_ERROR(
                        logdog::message="bwlist_client got response: (" + std::to_string(result.status) + ") " + result.reason,
                        logdog::where_name=BWLIST);
                    ec = EError::BWListError;
                    yield break;
                }

                ec = ParseLists(result);

                yield break;
            }
        } catch (const std::exception& e) {
            NSLS_LOG_CTX_ERROR(
                log::uid=Uid,
                logdog::message="bwlist exception",
                logdog::exception=e,
                logdog::where_name=BWLIST);
            ec = EError::DeliveryInternal;
            return Callback(ec, TListPtr());
        }

        if (yctx.is_complete()) {
            Callback(ec, BWList);
        }
    }

private:
    void TryExtractAddresses(const boost::property_tree::ptree& addrList, EType type) {
        for (const auto& val: addrList) {
            const auto rawAddr = val.second.get_value<std::string>();
            try {
                auto encodedAddr = EmailHelpers::idnaize(rawAddr);
                BWList->Insert(type, std::move(encodedAddr));
            } catch (const rfc2822ns::invalid_address&) {
                NSLS_LOG_CTX_WARN(
                    log::uid=Uid,
                    logdog::message="invalid bwlist address: <" + rawAddr + ">",
                    logdog::where_name=BWLIST);
            }
        }
    }

    TErrorCode ParseLists(const NHttp::TResponse& result) {
        try {
            using boost::property_tree::ptree;
            using boost::property_tree::json_parser::read_json;

            ptree pt;
            std::istringstream is(result.body);
            read_json(is, pt);

            TryExtractAddresses(pt.get_child("blacklist"), EType::Black);
            TryExtractAddresses(pt.get_child("whitelist"), EType::White);

            NSLS_LOG_CTX_DEBUG(
                log::uid=Uid,
                logdog::message=BWList->ToString(),
                logdog::where_name=BWLIST);
        } catch (const std::exception& e) {
            NSLS_LOG_CTX_ERROR(
                log::uid=Uid,
                logdog::message="bwlist exception",
                logdog::exception=e,
                logdog::where_name=BWLIST);
            return EError::BWListError;
        }
        return {};
    }

    TContextPtr Ctx;
    NHttp::TClientPtr HttpClient;
    const std::string& Uid;
    TCallback Callback;
    TListPtr BWList;
};

#include <yplatform/unyield.h>

} // namespace NDetail

void LoadLists(TContextPtr ctx, const std::string& uid, TCallback cb) {
    return LoadListsWithHttpClient(
        ctx,
        uid,
        NHttp::CreateClientByName(ctx, BWLIST, *ctx->GetConfig()->BWList, "ghttp"),
        std::move(cb));
}

void LoadListsWithHttpClient(TContextPtr ctx, const std::string& uid, NHttp::TClientPtr httpClient, TCallback cb) {
    auto requester = std::make_shared<NDetail::TRequester>(ctx, httpClient, uid, std::move(cb));
    yplatform::spawn(requester);
}

} // namespace NNotSoLiteSrv::NBlackWhiteList
