#include "config.h"

#include <vector>

namespace NNotSoLiteSrv {

namespace {

using Initializer = std::function<void(const yplatform::ptree&)>;

template <typename Config>
Initializer CreateInitializer(Config& config) {
    return [&config](const yplatform::ptree& pt) {
        config = std::make_unique<typename Config::element_type>(pt);
    };
}

} // namespace

TConfig::TConfig(const yplatform::ptree& ptree) {
    const std::vector<std::pair<std::string, Initializer>> dict{
        {"blackbox",            CreateInitializer(Blackbox)},
        {"bwlist",              CreateInitializer(BWList)},
        {"firstline",           CreateInitializer(Firstline)},
        {"furita",              CreateInitializer(Furita)},
        {"tupita",              CreateInitializer(Tupita)},
        {"session",             CreateInitializer(Session)},
        {"mdbsave",             CreateInitializer(MdbSave)},
        {"message",             CreateInitializer(Message)},
        {"meta_save_op",        CreateInitializer(MetaSaveOp)},
        {"msearch",             CreateInitializer(MSearch)},
        {"msettings",           CreateInitializer(MSettings)},
        {"mthr",                CreateInitializer(Mthr)},
        {"domain_rule_forward", CreateInitializer(DomainForward)},
        {"forward",             CreateInitializer(Forward)},
        {"autoreply",           CreateInitializer(AutoReply)},
        {"notify",              CreateInitializer(Notify)},
        {"meta",                CreateInitializer(Meta)},
        {"user_journal",        CreateInitializer(UserJournal)},
        {"related_uids",        CreateInitializer(RelatedUids)},
        {"recognizer",          CreateInitializer(Recognizer)}
    };

    for (const auto& [section, initializer]: dict) {
        initializer(ptree.get_child(section));
    }
}

TConfig::~TConfig() = default;

} // namespace NNotSoLiteSrv
