#include "related_uids.h"
#include <boost/tokenizer.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <fstream>

namespace NNotSoLiteSrv::NConfig {

TRelatedUids::TContainer LoadRelatedUidsFromFile(const std::string& filename) {
    std::ifstream ifs(filename);
    if (!ifs) {
        throw std::runtime_error("Failed to open related uids file: '" + filename + "'");
    }

    std::string line;
    TRelatedUids::TContainer ret;
    while (std::getline(ifs, line)) {
        boost::trim(line);

        std::vector<std::string> vec;
        boost::tokenizer<boost::char_separator<char>> tokenizer(line, boost::char_separator<char>{" ,:\t"});
        boost::transform(tokenizer, std::back_inserter(vec), [](const auto& str) { return boost::trim_copy(str); });
        if (vec.size() < 2) {
            // every line should consist of original uid and minimum one destination uid
            continue;
        }
        ret.emplace(vec.front(), TRelatedUids::TContainer::mapped_type(std::next(vec.begin()), vec.end()));
    }

    return ret;
}

TRelatedUids::TRelatedUids(const yplatform::ptree& ptree): TSmtpClient(ptree) {
    std::string mapFilename = ptree.get<std::string>("map_filename");
    if (!mapFilename.empty()) {
        RelatedUids = LoadRelatedUidsFromFile(mapFilename);
    }
}

const TRelatedUids::TContainer::mapped_type TRelatedUids::GetRelatedUids(const std::string& uid) const {
    if (auto it = RelatedUids.find(uid); it != RelatedUids.end()) {
        return it->second;
    }
    return {};
}

} // namespace NNotSoLiteSrv::NConfig
