#include "context.h"
#include <boost/asio/ip/host_name.hpp>
#include <util/random/random.h>
#include <memory>
#include <random>

namespace NNotSoLiteSrv {

constexpr size_t SESSION_ID_LENGTH = 10;
constexpr size_t ENVELOPE_ID_LENGTH = 8;

std::string GenerateId(size_t length) {
    constexpr char ALPHA_NUMERIC[] = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    constexpr uint32_t MAX_IDX = sizeof(ALPHA_NUMERIC) - 1;

    std::string id;
    id.reserve(length);
    std::generate_n(
        std::back_inserter(id),
        length,
        []() {
            auto idx = RandomNumber<uint32_t>(MAX_IDX);
            return ALPHA_NUMERIC[idx];
        });

    return id;
}

TContext::TContext(TConfigPtr cfg, const std::string& sessionId, const std::string& envelopeId)
    : Config(cfg)
{
    SetSessionId(sessionId);
    SetEnvelopeId(envelopeId);
}

void TContext::SetSessionId(const std::string& sessionId) {
    if (sessionId.empty()) {
        SessionId = GenerateId(SESSION_ID_LENGTH);
    } else {
        SessionId = sessionId;
    }
    UpdateFullSessionId();
}

void TContext::SetEnvelopeId(const std::string& envelopeId) {
    if (envelopeId.empty()) {
        EnvelopeId = GenerateId(ENVELOPE_ID_LENGTH);
    } else {
        EnvelopeId = envelopeId;
    }
    UpdateFullSessionId();
}

TTaskContextPtr TContext::GetTaskContext(const std::string& suffix) {
    std::string id = suffix.empty() ? GetFullSessionId() : GetFullSessionId() + "-" + suffix;
    return boost::make_shared<TTaskContext>(id);
}

} // namespace NNotSoLiteSrv
