#include "deliverer.h"
#include "server.h"

#include <mail/message_types/lib/message_types.h>
#include <mail/notsolitesrv/src/message/parser.h>
#include <mail/notsolitesrv/src/message/processor.h>
#include <mail/notsolitesrv/src/multiuser_processor.h>
#include <mail/notsolitesrv/src/smtp/meta.h>
#include <mail/notsolitesrv/src/new_emails/processor.h>
#include <mail/notsolitesrv/src/config/msearch.h>
#include <mail/notsolitesrv/src/msearch/client_impl.h>
#include <mail/notsolitesrv/src/msearch/types/request.h>
#include <mail/notsolitesrv/src/msearch/types/response.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/notsolitesrv/src/util/log.h>

#include <yplatform/find.h>
#include <yplatform/log.h>

#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/map.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/find_if.hpp>
#include <boost/range/algorithm/for_each.hpp>
#include <boost/asio/coroutine.hpp>
#include <util/generic/algorithm.h>
#include <util/string/cast.h>

#include <algorithm>
#include <set>

namespace NNotSoLiteSrv {

namespace {

const auto DO_NOT_UPDATE_OUTER_DATA{false};

}

TDeliverer::TDeliverer(
    boost::asio::io_context* io,
    TContextPtr ctx,
    NRules::TDomainRulesPtr domainRules,
    NMetaSaveOp::TMetaSaveOpPtr metaSaveOp,
    const std::string& stid,
    const TEnvelope& envelope,
    NUser::TStoragePtr userStorage,
    const NTimeTraits::TSystemTimePoint& deliveryStartMark,
    TMessageProcessor processor,
    NNewEmails::TProcessor::TAsyncSender newEmailsSender,
    TCallback cb
)
    : Io(io)
    , Ctx(ctx)
    , DomainRules(domainRules)
    , MetaSaveOp(std::move(metaSaveOp))
    , Stid(stid)
    , Envelope(envelope)
    , UserStorage(userStorage)
    , DeliveryStartMark(deliveryStartMark)
    , SyncPreProcessMessage(processor)
    , Callback(std::move(cb))
    , IsHttpSession(true)
    , AsyncSendNewEmails(std::move(newEmailsSender))
{}

TDeliverer::TDeliverer(
    boost::asio::io_context* io,
    TContextPtr ctx,
    NRules::TDomainRulesPtr domainRules,
    NMetaSaveOp::TMetaSaveOpPtr metaSaveOp,
    std::shared_ptr<std::string> strmsg,
    const TEnvelope& envelope,
    NUser::TStoragePtr userStorage,
    const NTimeTraits::TSystemTimePoint& deliveryStartMark,
    NNewEmails::TProcessor::TAsyncSender newEmailsSender,
    TCallback cb
)
    : Io(io)
    , Ctx(ctx)
    , DomainRules(domainRules)
    , MetaSaveOp(std::move(metaSaveOp))
    , StrMsg(strmsg)
    , Envelope(envelope)
    , UserStorage(userStorage)
    , DeliveryStartMark(deliveryStartMark)
    , Callback(std::move(cb))
    , AsyncSendNewEmails(std::move(newEmailsSender))
{}

#include <yplatform/yield.h>

#define MEASURE_DELAY(name) for (Timers[name].Start(); !Timers[name].Ready(); Timers[name].Stop())

void TDeliverer::operator()(TYieldCtx yctx, TErrorCode ec) {
    try {
        reenter(yctx) {
            if (IsHttpSession) {
                MEASURE_DELAY("get_from_mds") {
                    yield GetMessageFromMds(yctx);
                }

                if (ec) {
                    Result = ec;
                    yield break;
                }
            }

            MEASURE_DELAY("parse") {
                try {
                    Message = ParseMessage(*StrMsg, Ctx, IsHttpSession);
                } catch (const std::exception& e) {
                    Result = EError::MessageParse;
                    NSLS_LOG_CTX_ERROR(
                        logdog::message="Exception while message parsing",
                        logdog::exception=e,
                        logdog::where_name=Where);
                }
            }
            if (Result) {
                yield break;
            }

            MEASURE_DELAY("preprocess") {
                Preprocess(Ctx, Message, Envelope, UserStorage, IsHttpSession);
                Message->SetDeliveryStartMark(DeliveryStartMark);
                if (SyncPreProcessMessage) {
                    (*SyncPreProcessMessage)(Message);
                }
            }

            MEASURE_DELAY("resolve") {
                yield ProcessUsers(Ctx, UserProcessor, *UserStorage, Io->wrap(yctx));
            }

            if (ec) {
                NSLS_LOG_CTX_ERROR(
                    logdog::message="MultiUserProcessor error: " + ec.message(),
                    logdog::where_name=Where);
                Result = ec;
                yield break;
            }

            if (!CheckRecipientsStatus()) {
                yield break;
            }

            Result = Message->UpdateResolvedUsers(*UserStorage);
            if (Result) {
                yield break;
            }

            if (!CheckRecipientsStatus()) {
                yield break;
            }

            if (!IsHttpSession && !Message->IsMeta()) {
                MEASURE_DELAY("mulca") {
                    yield StoreMessageInMds(yctx);
                }

                if (ec) {
                    Result = ec;
                    yield break;
                }
            }

            MakeMetaSaveOpRequest(DO_NOT_UPDATE_OUTER_DATA);
            MEASURE_DELAY("domain_rules") {
                yield {
                    DomainRules->SetParams(Message, *MetaSaveOpRequest, boost::asio::bind_executor(
                        *Io, yctx));
                    boost::asio::post(*Io, [domainRules = std::move(DomainRules)]{
                        yplatform::spawn(domainRules);});
                }
            }

            MakeMetaSaveOpRequest();

            if (SubscriptionsEnabled() && PrepareProcessSubscriptions()) {
                MEASURE_DELAY("msettings") {
                    while (!ec && !UidsForOptInSubsCheck.empty()) {
                        yield ProcessOptInSubsSettings(
                            yctx,
                            UidsForOptInSubsCheck.extract(UidsForOptInSubsCheck.cbegin()).value()
                        );
                    }
                }

                if (ec) {
                    Result = ec;
                    yield break;
                }

                MEASURE_DELAY("msearch") {
                    yield ProcessSubscriptions(yctx);
                }

                if (ec) {
                    Result = ec;
                    yield break;
                }
            }

            MEASURE_DELAY("meta_save_op") {
                yield {
                    MetaSaveOp->SetOpParams(*MetaSaveOpRequest, boost::asio::bind_executor(*Io, yctx));
                    boost::asio::post(*Io, [metaSaveOp = std::move(MetaSaveOp)]{
                        yplatform::spawn(metaSaveOp);});
                }
            }

            if (ec) {
                Result = ec;
            }

            LogDomainRuleIds();
            LogFilterIds();

            if (ec || !CheckRecipientsStatus()) {
                if (!IsHttpSession && Message->IsMetaNeeded()) {
                    MEASURE_DELAY("meta") {
                        yield NSmtp::AsyncSendMeta(Ctx, Envelope, Message, *StrMsg, UserStorage,
                            Io->wrap(std::bind(&TDeliverer::OnSendMetaMail, this, yctx, std::placeholders::_1)));
                    }
                }
                yield break;
            }

            NewEmailsProcessors = NNewEmails::CreateSuitableProcessors(
                Ctx, AsyncSendNewEmails, *UserStorage, Message, Envelope.MailFrom, *Io, MSettingsClient
            );
            NewEmailsProcessorIter = NewEmailsProcessors.begin();
            MEASURE_DELAY("senders") {
                while (NewEmailsProcessorIter != NewEmailsProcessors.end()) {
                    yield std::shared_ptr<NNewEmails::TProcessor>(std::move(*NewEmailsProcessorIter))->Process(Io->wrap(yctx));
                    ++NewEmailsProcessorIter;
                }
            }

            MEASURE_DELAY("uj") {
                LogTskv();
            }

            yield break;
        }
    } catch (const std::exception& e) {
        NSLS_LOG_CTX_ERROR(
            logdog::message="internal server error",
            logdog::exception=e,
            logdog::where_name=Where);
        Result = EError::DeliveryInternal;
        return Callback(Result, Message);
    }

    if (yctx.is_complete()) {
        if (!Result) {
            LogMsgDelay();
        }
        Callback(Result, Message);
    }
}
#undef MEASURE_DELAY

#include <yplatform/unyield.h>

void TDeliverer::MakeMetaSaveOpRequest(bool updateOuterData) {
    MetaSaveOpRequest = std::make_shared<NMetaSaveOp::TRequest>(Ctx, Message, Envelope, *UserStorage,
        updateOuterData);
}

void TDeliverer::InitMulcagateClientIfNeeded() {
    if (!MulcagateClient) {
        MulcagateClient = yplatform::find<NMds::TClient, std::shared_ptr>("ymod_mds");
    }
}

void TDeliverer::StoreMessageInMds(TYieldCtx yctx) {
    std::string uid;
    int usersWithSharedStid = 0;
    UsersWithPersonalHeaders.clear();
    InitMulcagateClientIfNeeded();
    for (const auto& [login, user]: UserStorage->GetFilteredUsers(NUser::Found) | NUser::ForDelivery) {
        if (Message->HasPersonalHeaders(user.Uid)) {
            UsersWithPersonalHeaders.emplace(user.Uid, user.Suid);
        } else {
            uid = user.Uid;
            ++usersWithSharedStid;
        }
    }

    if (usersWithSharedStid == 0 && UsersWithPersonalHeaders.empty()) {
        return yctx(EError::DeliveryNoRecipients);
    }

    using namespace NMulcagate;
    NMulti::TRequests requests;
    for (const auto& [uid, suid]: UsersWithPersonalHeaders) {
        NMulti::TRequest req;
        req.Uid = uid;
        req.Spam = Message->IsSpam(uid);
        req.Data = Message->Compose(uid);
        requests.emplace(NMds::MakeStid(uid, false), req);
    }
    if (usersWithSharedStid > 0) {
        NMulti::TRequest req;
        req.Spam = Message->IsSpam();
        req.Data = Message->Compose();
        requests.emplace(NMds::MakeStid(uid, usersWithSharedStid > 1), req);
    }

    namespace ph = std::placeholders;
    return NMulti::Put(Ctx, requests, MulcagateClient, Io->wrap(std::bind(&TDeliverer::OnStoreMessageInMds, this, yctx, ph::_1, ph::_2)));
}

void TDeliverer::OnStoreMessageInMds(TYieldCtx yctx, TErrorCode ec, const NMulcagate::NMulti::TResults& res) {
    if (ec) {
        return yctx(ec);
    }
    for (const auto& [uid, putData]: res) {
        if (IsIn(UsersWithPersonalHeaders, uid)) {
            for (auto& [_, user]: UserStorage->GetFilteredUsers(NUser::Found) | NUser::ByUid(uid) | NUser::ForDelivery) {
                if (putData.ErrorCode) {
                    user.DeliveryResult.ErrorCode = putData.ErrorCode;
                } else {
                    Message->SetStid(putData.Stid, uid);
                }
            }
        } else {
            if (putData.ErrorCode) {
                for (auto& [_, user]: UserStorage->GetFilteredUsers(NUser::Found) | NUser::ForDelivery) {
                    if (!user.DeliveryResult.ErrorCode && !IsIn(UsersWithPersonalHeaders, user.Uid)) {
                        user.DeliveryResult.ErrorCode = putData.ErrorCode;
                    }
                }
            } else {
                Message->SetStid(putData.Stid);
            }
        }
    }

    if (Message->GetStid().empty()) {
        for (const auto& [uid, suid]: UsersWithPersonalHeaders) {
            auto stid = Message->GetPersonalStid(uid);
            if (stid) {
                Message->SetStid(*stid);
                break;
            }
        }
    }


    if (Message->GetStid().empty()) {
        NSLS_LOG_CTX_ERROR(
            logdog::message="Can't get stid from mulcagate",
            logdog::where_name=Where);
        return yctx(EError::StorageError);
    }

    return yctx(ec);
}

void TDeliverer::GetMessageFromMds(TYieldCtx yctx) {
    InitMulcagateClientIfNeeded();
    if (Stid.empty()) {
        return yctx(EError::MessageParse);
    }

    const std::string where{"STORAGE"};
    namespace ph = std::placeholders;
    return MulcagateClient->Get(Ctx->GetTaskContext(where), Stid, Io->wrap(std::bind(&TDeliverer::OnGetMessageFromMds, this, yctx, ph::_1, ph::_2)));
}

void TDeliverer::OnGetMessageFromMds(TYieldCtx yctx, TErrorCode ec, const std::string& message) {
    if (ec) {
        NSLS_LOG_CTX_ERROR(
            logdog::message="mulcagate error",
            logdog::error_code=ec,
            logdog::where_name=Where);
        return yctx(ec);
    }

    StrMsg = std::make_shared<std::string>(message);
    return yctx(ec);
}

void TDeliverer::OnSendMetaMail(TYieldCtx yctx, TErrorCode ec) {
    if (ec) {
        auto userWithPersonalHeaders = [msg=Message](const NUser::TUser& user) {
            return user.DeliveryParams.NeedDelivery &&
                user.DeliveryResult.ErrorCode &&
                !NError::IsPermError(user.DeliveryResult.ErrorCode) &&
                msg->HasPersonalHeaders(user.Uid) &&
                msg->GetPersonalStid(user.Uid) &&
                !msg->GetPersonalStid(user.Uid)->empty();
        };
        NUtil::LogStidForRemove(Ctx, Message->GetStid());
        for (const auto& [email, user]: UserStorage->GetFilteredUsers(NUser::Found) | userWithPersonalHeaders) {
            NUtil::LogStidForRemove(Ctx, Message->GetPersonalStid(user.Uid).get());
        }
    } else {
        Result = ec;
        std::set<std::string> stids;
        std::set<std::string> stidsToRemove;
        auto metaUser = [](const NUser::TUser& user) {
            return user.DeliveryResult.MetaSend;
        };
        for (const auto& [email, user]: UserStorage->GetFilteredUsers(NUser::Found) | metaUser) {
            auto stid = Message->GetStid(user.Uid);
            if (user.DeliveryResult.ErrorCode) {
                if (!IsIn(stids, stid)) {
                    stidsToRemove.emplace(Message->GetStid(user.Uid));
                }
            } else {
                stidsToRemove.erase(stid);
                stids.emplace(stid);
            }
        }
        for (const auto& stid: stids) {
            NSLS_LOG_CTX_NOTICE(
                logdog::message="meta message submit successful: " + stid,
                logdog::where_name=Where);
        }
        for (const auto& stid: stidsToRemove) {
            NUtil::LogStidForRemove(Ctx, stid);
        }
    }

    return yctx(ec);
}

bool TDeliverer::SubscriptionsEnabled() const {
    return !Ctx->GetConfig()->MSearch->MessageTypes.empty();
}

std::optional<std::string> TDeliverer::GetFrom() const {
    const auto& from = Message->GetFrom();
    return (from && !from->empty()) ? std::make_optional(from->front().second) : std::nullopt;
}

std::set<TUid> TDeliverer::GetUidsWithSubscriptions() const {
    using namespace boost::adaptors;
    auto uids = MetaSaveOpRequest->recipients
        | map_values
        | filtered([](const auto& rcpt) { return rcpt.params.process_as_subscription; })
        | transformed([](const auto& rcpt) { return rcpt.user.uid; });

    return { uids.begin(), uids.end() };
}

bool TDeliverer::PrepareProcessSubscriptions() {
    if (const auto& from = GetFrom(); !from || from->empty()) {
        NSLS_LOG_CTX_WARN(logdog::message="empty from address", logdog::where_name=Where);
        return false;
    }

    UidsForOptInSubsCheck = GetUidsWithSubscriptions();
    return !UidsForOptInSubsCheck.empty();
}

void TDeliverer::ProcessOptInSubsSettings(TYieldCtx yctx, TUid uid) {
    NMSettings::TParamsRequest request = {
        .Uid = uid,
        .Params = { "mail_b2c_can_use_opt_in_subs", "opt_in_subs_enabled" }
    };

    return MSettingsClient->GetParams(Ctx, request, [this, yctx, uid](auto ec, auto result) {
        OnProcessOptInSubsSettings(yctx, uid, ec, std::move(result));
    });
}

void TDeliverer::OnProcessOptInSubsSettings(TYieldCtx yctx, TUid uid, TErrorCode ec, NMSettings::TParamsResult result) {
    if (!ec && result) {
        if (result->CanUseOptInSubs.value_or(false) && result->OptInSubsEnabled.value_or(false)) {
            auto rcptIt = boost::range::find_if(MetaSaveOpRequest->recipients, [uid](const auto& item) {
                return item.second.user.uid == uid;
            });

            rcptIt->second.user.opt_in_subs_enabled = true;
        }
    }

    return yctx(ec);
}

void TDeliverer::ProcessSubscriptions(TYieldCtx yctx) {
    NMSearch::TSubscriptionStatusRequest request;

    request.Uids = GetUidsWithSubscriptions();
    if (request.Uids.empty()) {
        return yctx(EError::Ok);
    }

    using namespace boost::adaptors;
    auto optInSubsUids = MetaSaveOpRequest->recipients
        | map_values
        | filtered([](const auto& rcpt) { return rcpt.user.opt_in_subs_enabled.value_or(false); })
        | transformed([](const auto& rcpt) { return rcpt.user.uid; });

    request.OptInSubsUids = { optInSubsUids.begin(), optInSubsUids.end() };
    request.SubscriptionEmail = GetFrom().value();

    return MSearchClient->SubscriptionStatus(*Io, request, [this, yctx](auto ec, auto result) {
        OnProcessSubscriptions(yctx, ec, std::move(result));
    });
}

void TDeliverer::OnProcessSubscriptions(TYieldCtx yctx, TErrorCode ec, NMSearch::TSubscriptionStatusResult result) {
    if (!ec) {
        std::map<NMSearch::ESubscriptionStatus, std::set<TUid>> uidsByStatus;
        using namespace boost::adaptors;
        using namespace boost::algorithm;
        boost::range::for_each(
            result->Subscriptions | filtered([](const auto& status) {
                return status.Status == NMSearch::ESubscriptionStatus::hidden
                    || status.Status == NMSearch::ESubscriptionStatus::pending;
            }),
            [&](const auto& status) { uidsByStatus[status.Status].insert(status.Uid); }
        );

        if (const auto& uids = uidsByStatus[NMSearch::ESubscriptionStatus::pending]; !uids.empty()) {
            auto rcpts = MetaSaveOpRequest->recipients
                | map_values
                | filtered([&](const auto& rcpt) {
                    return rcpt.params.use_filters && !rcpt.params.spam && uids.contains(rcpt.user.uid);
                });

            if (!boost::empty(rcpts)) {
                NSLS_LOG_CTX_NOTICE(
                    logdog::message="mark move to pending by subscription status for uids: "
                        + join(rcpts | transformed([](const auto& rcpt) { return std::to_string(rcpt.user.uid); }), ", "),
                    logdog::where_name=Where
                );

                for (auto& rcpt : rcpts) {
                    auto& params = rcpt.params;
                    if (!params.folder) {
                        params.folder.emplace();
                    }
                    if (!params.folder->path) {
                        params.folder->path.emplace();
                    }
                    params.folder->path->path = "\\Pending";
                    params.no_such_folder_action = NMetaSaveOp::ENoSuchFolderAction::Create;
                    params.use_filters = false;
                }
            }
        }

        if (const auto& uids = uidsByStatus[NMSearch::ESubscriptionStatus::hidden]; !uids.empty()) {
            auto rcpts = MetaSaveOpRequest->recipients
                | map_values
                | filtered([&](const auto& rcpt) { return rcpt.params.use_filters && uids.contains(rcpt.user.uid); });

            if (!boost::empty(rcpts)) {
                NSLS_LOG_CTX_NOTICE(
                    logdog::message="mark move to trash by hidden subscription status for uids: "
                        + join(rcpts | transformed([](const auto& rcpt) { return std::to_string(rcpt.user.uid); }), ", "),
                    logdog::where_name=Where
                );

                for (auto& rcpt : rcpts) {
                    auto& params = rcpt.params;
                    if (!params.folder) {
                        params.folder.emplace();
                    }
                    if (!params.folder->path) {
                        params.folder->path.emplace();
                    }
                    params.folder->path->path = "\\Trash";
                    params.use_filters = false;
                }
            }
        }
    }

    return yctx(ec);
}

void TDeliverer::LogDomainRuleIds() const {
    const auto storageUsers{UserStorage->GetFilteredUsers(NUser::Found) | NUser::ForDelivery |
        NUser::WithDeliveryId};
    for (const auto& storageUser : storageUsers) {
        const auto& user{storageUser.second};
        if (user.DeliveryParams.UseDomainRules) {
            const std::string separator{","};
            NSLS_LOG_CTX_NOTICE(
                log::message="domain rules in use",
                log::org_id=*user.OrgId,
                log::uid=user.Uid,
                log::domain_rule_ids=boost::algorithm::join(user.DeliveryResult.DomainRuleIds, separator),
                log::where_name=Where);
        } else {
            NSLS_LOG_CTX_NOTICE(
                log::message="domain rules not in use",
                log::uid=user.Uid,
                log::where_name=Where);
        }
    }
}

void TDeliverer::LogFilterIds() const {
    const auto storageUsers{UserStorage->GetFilteredUsers(NUser::Found) | NUser::ForDelivery |
        NUser::WithDeliveryId};
    for (const auto& storageUser : storageUsers) {
        const auto& user{storageUser.second};
        if (MetaSaveOpRequest->recipients.at(user.DeliveryParams.DeliveryId).params.use_filters) {
            const std::string separator{","};
            NSLS_LOG_CTX_NOTICE(
                log::message="filters in use",
                log::uid=user.Uid,
                log::filter_ids=boost::algorithm::join(user.DeliveryResult.FilterIds, separator),
                log::where_name=Where);
        } else {
            NSLS_LOG_CTX_NOTICE(
                log::message="filters not in use",
                log::uid=user.Uid,
                log::where_name=Where);
        }
    }
}

void TDeliverer::LogTskv() {
    if (!UJWriter) {
        UJWriter = yplatform::find<IServer>("nsls")->GetUserJournalWriter();
    }

    for (const auto& userToLog: UserStorage->GetFilteredUsers(NUser::Found) | NUser::ForDelivery) {
        NTskv::LogAll(Ctx, UJWriter, Message, userToLog.second);
    }
}

void TDeliverer::LogMsgDelay() const {
    auto now = NTimeTraits::Now();
    const auto& users = UserStorage->GetFilteredUsers([](const NUser::TUser& user) {
        return user.DeliveryParams.IsFromRcptTo &&
            user.Status == NUser::ELoadStatus::Found &&
            !user.DeliveryResult.ErrorCode &&
            !user.DeliveryResult.MetaSend;
    });
    if (Message->GetStartMark() == NTimeTraits::FromUnixTime(0)) {
        // There is no X-Yandex-TimeMark header
        Message->SetStartMark(Message->GetDeliveryStartMark());
    }
    for (const auto& [email, rcpt]: users) {
        std::stringstream msg;
        msg
            << "msg_delay email=" << email << " mulcaid=" << Message->GetStid(rcpt.Uid)
            << " imap=" << (Message->GetXYHintByUid(rcpt.Uid).imap ? "true" : "false")
            << " in=" << NTimeTraits::ToString(Message->GetDeliveryStartMark() - Message->GetStartMark())
            << " out=" << NTimeTraits::ToString(now - Message->GetStartMark())
            << " dlv=" << NTimeTraits::ToString(now - Message->GetDeliveryStartMark())
            << " " << GetDelays();
        NSLS_LOG_CTX_NOTICE(logdog::message=msg.str(), logdog::where_name=Where);
    }
}

bool TDeliverer::CheckRecipientsStatus() {
    const auto& rcpts = UserStorage->GetUsers();

    CountOkRcpts = 0;
    for (const auto& rcpt: rcpts) {
        if (!rcpt.second.DeliveryParams.NeedDelivery) {
            continue;
        }
        if (rcpt.second.Status != NUser::ELoadStatus::Found) {
            continue;
        }
        if (rcpt.second.DeliveryResult.ErrorCode) {
            continue;
        }

        ++CountOkRcpts;
    }

    return CountOkRcpts != 0;
}

std::string TDeliverer::GetDelays() const {
    std::string ret;

    for (const auto& [name, timer]: Timers) {
        if (!ret.empty()) {
            ret.append(" ");
        }

        ret.append(name).append("=").append(NTimeTraits::ToString(timer.GetDuration()));
    }

    return ret;
}

} // namespace NNotSoLiteSrv
