#pragma once

#include "envelope.h"
#include "errors.h"
#include "multiuser_processor.h"
#include "timer.h"

#include <mail/notsolitesrv/src/http/client.h>
#include <mail/notsolitesrv/src/message/message.h>
#include <mail/notsolitesrv/src/meta_save_op/imeta_save_op.h>
#include <mail/notsolitesrv/src/meta_save_op/types/request.h>
#include <mail/notsolitesrv/src/msearch/client.h>
#include <mail/notsolitesrv/src/msettings/client.h>
#include <mail/notsolitesrv/src/mulcagate/client.h>
#include <mail/notsolitesrv/src/mulcagate/multiput.h>
#include <mail/notsolitesrv/src/new_emails/processor.h>
#include <mail/notsolitesrv/src/rules/domain/idomain_rules.h>
#include <mail/notsolitesrv/src/tskv/userjournal_writer.h>
#include <mail/notsolitesrv/src/user/storage.h>

#include <yplatform/coroutine.h>
#include <yplatform/util/ntimer.h>

#include <boost/asio/io_context.hpp>

#include <memory>
#include <optional>

namespace NNotSoLiteSrv {

class TDeliverer {
public:
    using TCallback = std::function<void(TErrorCode, TMessagePtr)>;
    using TMessageProcessor = std::function<void(TMessagePtr)>;
    using TYieldCtx = yplatform::yield_context<TDeliverer>;
    TDeliverer(
        boost::asio::io_context* io,
        TContextPtr ctx,
        NRules::TDomainRulesPtr domainRules,
        NMetaSaveOp::TMetaSaveOpPtr metaSaveOp,
        std::shared_ptr<std::string> strmsg,
        const TEnvelope& envelope,
        NUser::TStoragePtr userStorage,
        const NTimeTraits::TSystemTimePoint& deliveryStartMark,
        NNewEmails::TProcessor::TAsyncSender newEmailsSender,
        TCallback cb
    );

    TDeliverer(
        boost::asio::io_context* io,
        TContextPtr ctx,
        NRules::TDomainRulesPtr domainRules,
        NMetaSaveOp::TMetaSaveOpPtr metaSaveOp,
        const std::string& stid,
        const TEnvelope& envelope,
        NUser::TStoragePtr userStorage,
        const NTimeTraits::TSystemTimePoint& deliveryStartMark,
        TMessageProcessor processor,
        NNewEmails::TProcessor::TAsyncSender newEmailsSender,
        TCallback cb
    );

    void operator()(TYieldCtx yctx, TErrorCode ec = TErrorCode());

    void SetMultiUserProcessor(TMultiUserProcessor multiUserProcessor) {
        ProcessUsers = multiUserProcessor;
    }

    void SetUserProcessor(NUser::TProcessor userProcessor) {
        UserProcessor = userProcessor;
    }

    void SetMulcagateClient(NMulcagate::TClientPtr mulcagateClient) {
        MulcagateClient = mulcagateClient;
    }

    void SetMSearchClient(NMSearch::TMSearchClientPtr msearchClient) {
        MSearchClient = msearchClient;
    }

    void SetMSettingsClient(NMSettings::TMSettingsClientPtr msettingsClient) {
        MSettingsClient = std::move(msettingsClient);
    }

    void SetUserJournalWriter(NTskv::TUserJournalPtr ujWriter) {
        UJWriter = ujWriter;
    }

private:
    void MakeMetaSaveOpRequest(bool updateOuterData = true);
    void InitMulcagateClientIfNeeded();

    void StoreMessageInMds(TYieldCtx yctx);
    void OnStoreMessageInMds(TYieldCtx yctx, TErrorCode ec, const NMulcagate::NMulti::TResults& res);

    void GetMessageFromMds(TYieldCtx yctx);
    void OnGetMessageFromMds(TYieldCtx yctx, TErrorCode ec, const std::string& message);

    void OnSendMetaMail(TYieldCtx yctx, TErrorCode ec);

    bool SubscriptionsEnabled() const;
    std::optional<std::string> GetFrom() const;
    std::set<TUid> GetUidsWithSubscriptions() const;
    bool PrepareProcessSubscriptions();
    void ProcessOptInSubsSettings(TYieldCtx yctx, TUid uid);
    void OnProcessOptInSubsSettings(TYieldCtx yctx, TUid uid, TErrorCode ec, NMSettings::TParamsResult result);
    void ProcessSubscriptions(TYieldCtx yctx);
    void OnProcessSubscriptions(TYieldCtx yctx, TErrorCode ec, NMSearch::TSubscriptionStatusResult result);

    void LogDomainRuleIds() const;
    void LogFilterIds() const;
    void LogTskv();
    void LogMsgDelay() const;

    bool CheckRecipientsStatus();
    std::string GetDelays() const;

private:
    boost::asio::io_context* Io = nullptr;
    TContextPtr Ctx;
    NRules::TDomainRulesPtr DomainRules;
    NMetaSaveOp::TMetaSaveOpPtr MetaSaveOp;
    std::string Stid;
    std::shared_ptr<std::string> StrMsg;
    TEnvelope Envelope;
    NUser::TStoragePtr UserStorage;
    NTimeTraits::TSystemTimePoint DeliveryStartMark;
    TMessagePtr Message;
    std::optional<TMessageProcessor> SyncPreProcessMessage;
    TCallback Callback;
    bool IsHttpSession = false;
    TMultiUserProcessor ProcessUsers = ProcessAllUsers;
    NUser::TProcessor UserProcessor = NUser::ProcessUser;
    NMulcagate::TClientPtr MulcagateClient;
    NMSearch::TMSearchClientPtr MSearchClient;
    NMSettings::TMSettingsClientPtr MSettingsClient;
    NNewEmails::TProcessor::TAsyncSender AsyncSendNewEmails;
    NNewEmails::TProcessors NewEmailsProcessors;
    NNewEmails::TProcessors::iterator NewEmailsProcessorIter;
    NTskv::TUserJournalPtr UJWriter;
    std::unordered_map<std::string, TTimer> Timers;
    std::map<std::string, std::string> UsersWithPersonalHeaders; // uid, suid
    std::set<TUid> UidsForOptInSubsCheck;
    const std::string Where {"DLV"};
    std::shared_ptr<NMetaSaveOp::TRequest> MetaSaveOpRequest;
    TErrorCode Result;
    size_t CountOkRcpts = 0;
};

using TDelivererPtr = std::shared_ptr<TDeliverer>;

} // namespace NNotSoLiteSrv
