#pragma once

#include <boost/system/error_code.hpp>

namespace NNotSoLiteSrv {

using TErrorCode = ::boost::system::error_code;
using TErrorCategory = ::boost::system::error_category;

enum class EError {
    Ok = 0,
    MessageParse,
    UserNotFound,
    UserInvalid,
    BWListError,
    BWListBlacklisted,
    StorageError,
    StorageMailNotFound,
    MetaSaveOpError,
    MetaSaveOpPermanentError,
    MetaSaveOpTemporaryError,
    MSearchError,
    MetaSaveOpIncorrectResult,
    MultiUserError,
    SmtpTemporaryError,
    SmtpPermanentError,
    SmtpUnknownError,
    DeliveryNoRecipients,
    DeliveryTaskCancelled,
    DeliveryLoopDetected,
    DeliveryCorruptedParams,
    DeliveryInternal,
    DomainRulesIncorrectResult,
    HttpRequestParseError,
    HttpNonRetryableStatus,
    HttpRetriesExceeded,
    FirstlineError,
    FuritaOrgNotFound,
    FuritaResponseParseError,
    MdbSaveResponseParseError,
    MSearchResponseParseError,
    MSettingsResponseParseError,
    MthrError,
    TupitaResponseParseError,
    TvmServiceTicketError,
    NotImplemented
};

const TErrorCategory& category() noexcept;

} // namespace NNotSoLiteSrv

namespace boost {
namespace system {

template <>
struct is_error_code_enum<NNotSoLiteSrv::EError> : public std::true_type {};

} // namespace system
} // namespace boost

namespace NNotSoLiteSrv {

inline auto make_error_code(const EError e) {
    return boost::system::error_code(static_cast<int>(e), category());
}

class TCategory : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "NNotSoLiteSrv::NError::TCategory";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
        case EError::Ok:
            return "ok";
        case EError::MessageParse:
            return "message parse error";
        case EError::UserNotFound:
            return "user not found";
        case EError::UserInvalid:
            return "invalid user";
        case EError::BWListError:
            return "black/white list error";
        case EError::BWListBlacklisted:
            return "blacklisted";
        case EError::StorageError:
            return "storage error";
        case EError::StorageMailNotFound:
            return "message not found in the storage";
        case EError::DeliveryNoRecipients:
            return "empty recipients list";
        case EError::DeliveryLoopDetected:
            return "loop detected";
        case EError::DeliveryInternal:
            return "internal error";
        case EError::DeliveryCorruptedParams:
            return "incorrect delivery parameters";
        case EError::DeliveryTaskCancelled:
            return "task is cancelled";
        case EError::DomainRulesIncorrectResult:
            return "domain rules result is incorrect";
        case EError::MetaSaveOpError:
            return "meta_save_op error";
        case EError::MetaSaveOpPermanentError:
            return "permanent error";
        case EError::MetaSaveOpTemporaryError:
            return "temporary error";
        case EError::MSearchError:
            return "msearch error";
        case EError::MdbSaveResponseParseError:
            return "mdbsave response parse error";
        case EError::MetaSaveOpIncorrectResult:
            return "meta save op result is incorrect";
        case EError::MSearchResponseParseError:
            return "msearch response parse error";
        case EError::MSettingsResponseParseError:
            return "msettings response parse error";
        case EError::MultiUserError:
            return "multiple users processor error";
        case EError::SmtpTemporaryError:
            return "remote smtp server responds with temporary error";
        case EError::SmtpPermanentError:
            return "remote smtp server responds with permanent error";
        case EError::SmtpUnknownError:
            return "unknown remote smtp server error";
        case EError::HttpRequestParseError:
            return "could not parse request";
        case EError::HttpNonRetryableStatus:
            return "status is not retryable";
        case EError::HttpRetriesExceeded:
            return "retries exceeded";
        case EError::FirstlineError:
            return "firstline error";
        case EError::FuritaOrgNotFound:
            return "organization not found in furita";
        case EError::FuritaResponseParseError:
            return "furita response parse error";
        case EError::MthrError:
            return "mthr error";
        case EError::TupitaResponseParseError:
            return "tupita response parse error";
        case EError::TvmServiceTicketError:
            return "failed to get tvm service ticket";
        case EError::NotImplemented:
            return "not implemented yet";
        }

        return "unknown error (" + std::to_string(value) + ")";
    }
};

inline const TErrorCategory& category() noexcept {
    static TCategory instance;
    return instance;
}

namespace NError {

inline bool IsPermError(const EError e) {
    switch (e) {
    case EError::MetaSaveOpPermanentError:
    case EError::DeliveryNoRecipients:
    case EError::DeliveryLoopDetected:
    case EError::DeliveryCorruptedParams:
    case EError::DomainRulesIncorrectResult:
    case EError::MSearchResponseParseError:
    case EError::MSettingsResponseParseError:
    case EError::MessageParse:
    case EError::MetaSaveOpIncorrectResult:
    case EError::UserNotFound:
    case EError::UserInvalid:
    case EError::HttpRequestParseError:
    case EError::FirstlineError:
    case EError::FuritaOrgNotFound:
    case EError::FuritaResponseParseError:
    case EError::MthrError:
    case EError::TupitaResponseParseError:
    case EError::MdbSaveResponseParseError:
    case EError::TvmServiceTicketError:
    case EError::NotImplemented:
        return true;

    case EError::Ok:
    case EError::MetaSaveOpError:
    case EError::MetaSaveOpTemporaryError:
    case EError::MSearchError:
    case EError::BWListError:
    case EError::BWListBlacklisted:
    case EError::DeliveryInternal:
    case EError::DeliveryTaskCancelled:
    case EError::MultiUserError:
    case EError::StorageError:
    case EError::StorageMailNotFound:
    case EError::SmtpTemporaryError:
    case EError::SmtpPermanentError:
    case EError::SmtpUnknownError:
    case EError::HttpNonRetryableStatus:
    case EError::HttpRetriesExceeded:
        return false;
    }

    return false;
}

inline bool IsPermError(const TErrorCode ec) {
    if (ec.category() == category()) {
        return IsPermError(static_cast<EError>(ec.value()));
    }

    return false;
}

inline bool IsOk(const EError e) {
    return e == EError::Ok || e == EError::DeliveryLoopDetected;
}

inline bool IsOk(const TErrorCode ec) {
    if (!ec) {
        return true;
    }
    if (ec.category() == category()) {
        return IsOk(static_cast<EError>(ec.value()));
    }

    return false;
}

} // namespace NError
} // namespace NNotSoLiteSrv
