#include "firstline_impl.h"

#include <mail/notsolitesrv/lib/firstline/first_line.h>
#include <mail/notsolitesrv/lib/firstline/message_text.h>
#include <mail/notsolitesrv/lib/firstline/legacy_first_line.h>
#include <mail/notsolitesrv/lib/firstline/subscription.h>
#include <mail/notsolitesrv/lib/firstline/subscription_source.h>
#include <mail/notsolitesrv/lib/firstline/types.h>
#include <mail/notsolitesrv/src/util/file.h>
#include <mail/library/utf8/utf8.h>

#include <util/string/join.h>

#include <boost/algorithm/string.hpp>

namespace NNotSoLiteSrv::NFirstline::NImpl {

namespace {

NLib::TSubscriptionSourcePtr InitializeSubscriptionSource(const std::string& subscriptionRules) {
    auto source = NLib::CreateSubscriptionSource(subscriptionRules.data(), subscriptionRules.size());
    if (!source) {
        throw std::runtime_error("Unable to initialize subscription source");
    }

    return std::move(source);
}

NLib::TLegacyFirstLinePtr InitializeLegacyFirstLine(const std::string& firstLineRules) {
    auto firstline = std::make_shared<NLib::CFirstLine>();
    std::stringstream ss(firstLineRules);
    if (!firstline->FL_LoadRules(ss)) {
        throw std::runtime_error("Unable to initialize legacy firstline");
    }

    return firstline;
}

std::string BuildAddrSpec(const std::string& local, const std::string& domain) {
    return Join('@', local, domain);
}

std::unique_ptr<const NLib::Subscription> FindSubscription(
    const NLib::SubscriptionSource* subscriptionSource,
    const std::string& addrSpec,
    const std::string& displayName,
    const std::string& subject)
{
    if (!subscriptionSource) {
        return {};
    }

    return NLib::CreateSubscription(addrSpec, displayName, subject, subscriptionSource);
}

}

TFirstline::TFirstline(const NConfig::TFirstlineUptr& config)
    : FirstLineMaxSize{config->MaxSize}
    , LegacyFirstLine{InitializeLegacyFirstLine(NUtil::ReadFile(config->LegacyFirstLineRules))}
    , SubscriptionSource{InitializeSubscriptionSource(NUtil::ReadFile(config->SubscriptionRules))}
{
}

std::string TFirstline::GenerateFirstline(const TFirstlineRequest& request) const {
    NLib::MsgText msgText;
    auto part = ::NUtil::Utf8Sanitized(request.Part);
    if (request.IsHtml) {
        msgText.fromHtml(part.c_str(), request.IsPeopleType);
    } else {
        msgText.fromText(part.c_str());
    }

    auto subscription = FindSubscription(
        SubscriptionSource.get(),
        BuildAddrSpec(request.From.Local, request.From.Domain),
        request.From.DisplayName,
        request.Subject);

    std::string firstline;
    auto result = NLib::FindFirstLine(msgText, subscription.get(), firstline,
        FirstLineMaxSize, request.IsPeopleType);
    const auto errorCode = -1;
    if (result == errorCode) {
        auto buffer = std::make_unique<char[]>(FirstLineMaxSize);
        result = LegacyFirstLine->FL_Find(msgText.text(), buffer.get(), FirstLineMaxSize);
        firstline = buffer.get();
    }

    return boost::trim_copy(firstline);
}

} // namespace NNotSoLiteSrv::NFirstline::NImpl
