#include "client.h"

#include <mail/notsolitesrv/src/furita/types/reflection/response.h>

#include <yamail/data/deserialization/yajl.h>

#include <boost/asio.hpp>

#include <exception>
#include <string>
#include <utility>

namespace NNotSoLiteSrv::NFurita {

static const std::string FURITA{"FURITA"};

TFuritaClient::TFuritaClient(TContextPtr ctx, TClusterCallPtr clusterCall, TTvmModulePtr tvmModule)
    : Ctx{std::move(ctx)}
    , ClusterCall{std::move(clusterCall)}
    , TvmModule{std::move(tvmModule)}
{
}

void TFuritaClient::List(boost::asio::io_context& ioContext, TUid uid, TListCallback callback) const {
    auto clusterCallCallback = [=, executor = ioContext.get_executor()](auto&& errorCode, auto&& response) {
        boost::asio::post(executor, [=]{ProcessListResponse(std::move(errorCode), std::move(response),
            std::move(callback));});
    };

    ClusterCall->async_run(Ctx->GetTaskContext(FURITA), MakeListRequest(uid), std::move(clusterCallCallback));
}

void TFuritaClient::Get(boost::asio::io_context& ioContext, TOrgId orgId, TGetCallback callback) const {
    auto clusterCallCallback = [=, executor = ioContext.get_executor()](auto&& errorCode, auto&& response) {
        boost::asio::post(executor, [=]{ProcessGetResponse(std::move(errorCode), std::move(response),
            std::move(callback));});
    };

    try {
        ClusterCall->async_run(Ctx->GetTaskContext(FURITA), MakeGetRequest(std::move(orgId)), std::move(
            clusterCallCallback));
    } catch (EError error) {
        boost::asio::post(ioContext, [=]{callback(error, {});});
    }
}

yhttp::request TFuritaClient::MakeListRequest(TUid uid) const {
    return yhttp::request::GET("/api/list.json?uid=" + std::to_string(uid) + "&detailed=1");
}

void TFuritaClient::ProcessListResponse(TErrorCode errorCode, THttpResponse response,
    TListCallback callback) const
{
    if (errorCode || (response.status != 200)) {
        ProcessListError(std::move(errorCode), std::move(response), std::move(callback));
    } else {
        auto listResult{ParseListResponse(std::move(response.body))};
        if (!listResult) {
            errorCode = EError::FuritaResponseParseError;
        }

        callback(std::move(errorCode), std::move(listResult));
    }
}

TListResult TFuritaClient::ParseListResponse(const std::string responseBody) const {
    try {
        return yamail::data::deserialization::fromJson<TFuritaListResponse>(responseBody);
    } catch (const std::exception& ex) {
        NSLS_LOG_CTX_ERROR(log::message = "failed to parse Furita list.json response", log::exception = ex,
            log::where_name = FURITA);
        return {};
    }
}

yhttp::request TFuritaClient::MakeGetRequest(TOrgId orgId) const {
    auto url{"/v1/domain/rules/get?orgid=" + std::move(orgId)};
    if (Ctx->GetConfig()->Furita->UseTvm) {
        return THttpRequest::GET(std::move(url), MakeTvmServiceTicketHeader(GetTvmServiceTicket()));
    } else {
        return THttpRequest::GET(std::move(url));
    }
}

void TFuritaClient::ProcessGetResponse(TErrorCode errorCode, THttpResponse response,
    TGetCallback callback) const
{
    if (errorCode || (response.status != 200)) {
        ProcessGetError(std::move(errorCode), std::move(response), std::move(callback));
    } else {
        auto getResult{ParseGetResponse(std::move(response.body))};
        if (!getResult) {
            errorCode = EError::FuritaResponseParseError;
        }

        callback(std::move(errorCode), std::move(getResult));
    }
}

TGetResult TFuritaClient::ParseGetResponse(std::string responseBody) const
{
    try {
        return yamail::data::deserialization::fromJson<TFuritaGetResponse>(responseBody);
    } catch (const std::exception& ex) {
        NSLS_LOG_CTX_ERROR(log::message = "failed to parse Furita get response", log::exception = ex,
            log::where_name = FURITA);
        return {};
    }
}

std::string TFuritaClient::GetTvmServiceTicket() const {
    std::string ticket;
    const auto errorCode{TvmModule->get_service_ticket(Ctx->GetConfig()->Furita->Service, ticket)};
    if (errorCode) {
        NSLS_LOG_CTX_ERROR(log::error_code = errorCode, log::where_name = FURITA);
        throw EError::TvmServiceTicketError;
    }

    return ticket;
}

void TFuritaClient::ProcessListError(TErrorCode errorCode, THttpResponse response,
    TListCallback callback) const
{
    if (errorCode) {
        NSLS_LOG_CTX_ERROR(log::error_code = errorCode, log::where_name = FURITA);
    } else if ((response.status / 100) != 5) {
        errorCode = EError::HttpNonRetryableStatus;
        NSLS_LOG_CTX_ERROR(log::message = MakeHttpStatusMessage(std::move(response)),
            log::error_code = errorCode, log::where_name = FURITA);
    } else {
        errorCode = EError::HttpRetriesExceeded;
        NSLS_LOG_CTX_ERROR(log::message = MakeHttpStatusMessage(std::move(response)),
            log::error_code = errorCode, log::where_name = FURITA);
    }

    callback(std::move(errorCode), {});
}

void TFuritaClient::ProcessGetError(TErrorCode errorCode, THttpResponse response, TGetCallback callback) const
{
    if (errorCode) {
        NSLS_LOG_CTX_ERROR(log::error_code = errorCode, log::where_name = FURITA);
    } else if ((response.status / 100) != 5) {
        const auto furitaOrgNotFound{404};
        errorCode = (response.status == furitaOrgNotFound) ? EError::FuritaOrgNotFound :
            EError::HttpNonRetryableStatus;
        NSLS_LOG_CTX_ERROR(log::message = MakeHttpStatusMessage(std::move(response)),
            log::error_code = errorCode, log::where_name = FURITA);
    } else {
        errorCode = EError::HttpRetriesExceeded;
        NSLS_LOG_CTX_ERROR(log::message = MakeHttpStatusMessage(std::move(response)),
            log::error_code = errorCode, log::where_name = FURITA);
    }

    callback(std::move(errorCode), {});
}

}
