#include "client.h"
#include <mail/notsolitesrv/src/context.h>
#include <mail/notsolitesrv/src/errors.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/library/utf8/utf8.h>
#include <ymod_httpclient/call.h>
#include <yplatform/coroutine.h>
#include <yplatform/log.h>
#include <yplatform/find.h>
#include <yplatform/util/shared_ptr_cast.h>

namespace NNotSoLiteSrv::NHttp {

namespace NDetail {

#include <yplatform/yield.h>

using THttpClientPtr = std::shared_ptr<yhttp::call>;
using TTvm2ClientPtr = std::shared_ptr<ymod_tvm::tvm2_module>;

class TRequester : public std::enable_shared_from_this<TRequester> {
public:
    using TYieldCtx = yplatform::yield_context<TRequester>;
    using EMethod = yhttp::request::method_t;

    TRequester(
        TContextPtr ctx,
        const std::string& service,
        int attempts,
        THttpClientPtr httpClient,
        const yhttp::options& opts,
        TTvm2ClientPtr tvm2Client
    )
        : Ctx(ctx)
        , Service(service)
        , HttpClient(httpClient)
        , Options(opts)
        , Tvm2Client(tvm2Client)
        , TotalAttempts(attempts)
    {}

    void Start(
        EMethod method,
        const std::string& url,
        const THeaders& headers,
        const std::string& body,
        boost::optional<std::string> tvmServiceName,
        IClient::TCallback cb)
    {
        Method = method;
        Url = url;
        Headers = headers;
        Body = body;
        TvmServiceName = tvmServiceName;
        Callback = std::move(cb);

        yplatform::spawn(shared_from_this());
    }

    void operator()(
        TYieldCtx yctx,
        TErrorCode ec = TErrorCode(),
        yhttp::response response = yhttp::response())
    {
        try {
            reenter (yctx) {
                if (TvmServiceName && !TvmServiceName->empty()) {
                    if (!Tvm2Client) {
                        Tvm2Client = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("ymod_tvm");
                    }
                    std::string tvmServiceTicket;
                    auto err = Tvm2Client->get_service_ticket(*TvmServiceName, tvmServiceTicket);
                    if (err) {
                        throw std::runtime_error("failed to get service ticket for " + *TvmServiceName + ": " + err.message());
                    }
                    TvmServiceTicket.emplace(std::move(tvmServiceTicket));
                }

                for (; CurrentAttempt <= TotalAttempts; ++CurrentAttempt) {
                    yield HttpClient->async_run(
                        shared_ptr_cast<boost::shared_ptr>::from(Ctx->GetTaskContext(Service)),
                        MakeRequest(),
                        Options,
                        yctx);

                    if (ec) {
                        LogError(ec, response);
                        continue;
                    }

                    if (response.status / 100 != 2) {
                        LogError(ec, response);
                        if (response.status / 100 == 5) {
                            continue;
                        }
                    }

                    yield break;
                }
            }
        } catch (const std::exception& e) {
            NSLS_LOG_CTX_ERROR(
                logdog::message="Http client got exception",
                logdog::exception=e);
            ec = EError::DeliveryInternal;
            return Callback(ec, response);
        }

        if (yctx.is_complete()) {
            Callback(ec, response);
        }
    }

private:
    yhttp::request MakeRequest() const {
        using TRequest = yhttp::request;
        switch (Method) {
            case EMethod::GET:
                return TRequest::GET(Url, MakeHeaders());
            case EMethod::HEAD:
                return TRequest::HEAD(Url, MakeHeaders());
            case EMethod::POST:
                return TRequest::POST(Url, MakeHeaders(), std::string(Body));
            case EMethod::PUT:
                return TRequest::PUT(Url, MakeHeaders(), std::string(Body));
            case EMethod::DELETE:
                return TRequest::DELETE(Url, MakeHeaders());
        }
    }

    std::string MakeHeaders() const {
        std::string result;

        for (const auto& hdr: Headers) {
            result.append(hdr.first).append(": ").append(hdr.second).append("\r\n");
        }

        if (TvmServiceTicket && !TvmServiceTicket->empty()) {
            result.append("X-Ya-Service-Ticket: ").append(*TvmServiceTicket).append("\r\n");
        }

        return result;
    }

    void LogError(TErrorCode ec, const yhttp::response& response) {
        if (ec) {
            NSLS_LOG_CTX_ERROR(logdog::message="http client error: " + ec.message());
        } else {
            NSLS_LOG_CTX_ERROR(
                logdog::message=
                    "http client got response: (" + std::to_string(response.status) + ") " + response.reason +
                    ", body: " + ::NUtil::Utf8ByteHead(response.body, 1024));
        }
    }

private:
    TContextPtr Ctx;
    std::string Service;
    EMethod Method = EMethod::GET;
    std::string Url;
    THeaders Headers;
    std::string Body;
    IClient::TCallback Callback;

    boost::optional<std::string> TvmServiceName;
    boost::optional<std::string> TvmServiceTicket;

    THttpClientPtr HttpClient;
    yhttp::options Options;
    TTvm2ClientPtr Tvm2Client;

    int CurrentAttempt = 1;
    int TotalAttempts = 2;
};

#include <yplatform/unyield.h>

template <typename... TArgs>
auto MakeRequester(TArgs&&... args) {
    return std::make_shared<TRequester>(std::forward<TArgs>(args)...);
}

class TClient
    : public IClient
    , public std::enable_shared_from_this<TClient>
{
public:
    TClient(
        TContextPtr ctx,
        const std::string service,
        const NConfig::THttpCall& cfg,
        THttpClientPtr httpClient,
        TTvm2ClientPtr tvm2Client = TTvm2ClientPtr()
    )
        : Ctx(ctx)
        , Service(service)
        , Config(cfg)
        , HttpClient(httpClient)
        , Tvm2Client(tvm2Client)
        , TvmServiceName(cfg.TvmServiceName)
    {
        MakeOptions();
    }

    void Get(const std::string& url, const THeaders& headers, TCallback cb) override {
        auto requester = MakeRequester(Ctx, Service, Config.Attempts, HttpClient, Options, Tvm2Client);
        requester->Start(
            TRequester::EMethod::GET,
            url,
            MakeHeaders(headers),
            "",
            TvmServiceName,
            std::move(cb));
    }

    void Get(const std::string& url, TCallback cb) override {
        Get(url, THeaders(), std::move(cb));
    }

    void Head(const std::string& url, const THeaders& headers, TCallback cb) override {
        auto requester = MakeRequester(Ctx, Service, Config.Attempts, HttpClient, Options, Tvm2Client);
        requester->Start(
            TRequester::EMethod::HEAD,
            url,
            MakeHeaders(headers),
            "",
            TvmServiceName,
            std::move(cb));
    }

    void Head(const std::string& url, TCallback cb) override {
        Head(url, THeaders(), std::move(cb));
    }

    void Post(
        const std::string& url,
        const std::string& body,
        const THeaders& headers,
        TCallback cb) override
    {
        auto requester = MakeRequester(Ctx, Service, Config.Attempts, HttpClient, Options, Tvm2Client);
        requester->Start(
            TRequester::EMethod::POST,
            url,
            MakeHeaders(headers),
            body,
            TvmServiceName,
            std::move(cb));
    }

    void Post(
        const std::string& url,
        const std::string& body,
        TCallback cb) override
    {
        Post(url, body, THeaders(), std::move(cb));
    }

    void Post(
        const std::string& url,
        const yplatform::zerocopy::segment& zcBody,
        const THeaders& headers,
        TCallback cb) override
    {
        Post(url, std::string(zcBody.begin(), zcBody.end()), MakeHeaders(headers), std::move(cb));
    }

    void Post(
        const std::string& url,
        const yplatform::zerocopy::segment& zcBody,
        TCallback cb) override
    {
        Post(url, zcBody, THeaders(), std::move(cb));
    }

private:
    void MakeOptions() {
        Options.reuse_connection = Config.ReuseConnection;
        if (Config.TotalTimeout) {
            Options.timeouts.total = *Config.TotalTimeout;
        }
        if (Config.ConnectTimeout) {
            Options.timeouts.connect = *Config.ConnectTimeout;
        }
    }

    THeaders MakeHeaders(const THeaders& headers) const {
        THeaders requestHeaders;
        std::copy(Config.AddHeaders.cbegin(), Config.AddHeaders.cend(),
            std::inserter(requestHeaders, requestHeaders.end()));
        std::copy(headers.cbegin(), headers.cend(),
            std::inserter(requestHeaders, requestHeaders.end()));

        return requestHeaders;
    }

    TContextPtr Ctx;
    std::string Service;
    const NConfig::THttpCall& Config;
    THttpClientPtr HttpClient;
    yhttp::options Options;
    TTvm2ClientPtr Tvm2Client;
    boost::optional<std::string> TvmServiceName;
};

} // namespace NDetail

TClientPtr CreateClient(
    TContextPtr ctx,
    const std::string& service,
    const NConfig::THttpCall& cfg,
    std::shared_ptr<yhttp::call> yhttp,
    std::shared_ptr<ymod_tvm::tvm2_module> ytvm)
{
    return std::make_shared<NDetail::TClient>(ctx, service, cfg, yhttp, ytvm);
}

TClientPtr CreateClientByName(
    TContextPtr ctx,
    const std::string& service,
    const NConfig::THttpCall& cfg,
    const std::string& name)
{
    return std::make_shared<NDetail::TClient>(
        ctx,
        service,
        cfg,
        yplatform::find<yhttp::call, std::shared_ptr>(name));
}

void AppendUriParams(std::string& uri, const std::string& paramsStr) {
    if (paramsStr.empty()) {
        return;
    }

    uri.reserve(uri.size() + paramsStr.size() + 1);

    if (uri.rfind('?') == std::string::npos && uri.rfind('&') == std::string::npos) {
        uri.append("?");
    } else if (uri.back() != '?' && uri.back() != '&') {
        uri.append("&");
    }

    uri.append(paramsStr);
}

} // namespace NNotSoLiteSrv::NHttp
