#include "store.h"

#include "new_emails.h"

#include <mail/notsolitesrv/src/envelope.h>
#include <mail/notsolitesrv/src/http/types/reflection/response.h>
#include <mail/notsolitesrv/src/server.h>
#include <mail/notsolitesrv/src/message/message.h>
#include <mail/notsolitesrv/src/reflection/xyandexhint.h>
#include <mail/notsolitesrv/src/smtp/sender.h>
#include <mail/notsolitesrv/src/user/storage.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/notsolitesrv/src/types/time.h>

#include <util/generic/algorithm.h>
#include <util/generic/is_in.h>
#include <yamail/data/serialization/yajl.h>
#include <yplatform/coroutine.h>
#include <yplatform/find.h>
#include <yplatform/log/contains_logger.h>

namespace NNotSoLiteSrv::NHttp::NImpl {

namespace NDetail {

#include <yplatform/yield.h>

class TNewEmailsProcessor {
public:
    TNewEmailsProcessor(TContextPtr ctx, const NNewEmails::TDataProvider& dataProvider,
        TNewRecipients& newRecipients, NNewEmails::TProcessor::TSenderCallback callback)
        : Ctx(std::move(ctx))
        , DataProvider(dataProvider)
        , NewRecipients(newRecipients)
        , Callback(std::move(callback))
    {
    }

    using TYieldCtx = yplatform::yield_context<TNewEmailsProcessor>;
    void operator()(TYieldCtx yieldCtx, TErrorCode errorCode = {}) {
        reenter(yieldCtx) {
            yield AsyncFillResponseWithNewEmails(Ctx, DataProvider, NewRecipients, yieldCtx);
            if (errorCode) {
                yield break;
            }

            yield NSmtp::AsyncSmtpSender(Ctx, DataProvider, yieldCtx);
        }

        if (yieldCtx.is_complete()) {
            Callback(std::move(errorCode));
        }
    }

private:
    TContextPtr Ctx;
    const NNewEmails::TDataProvider& DataProvider;
    TNewRecipients& NewRecipients;
    NNewEmails::TProcessor::TSenderCallback Callback;
};

class TStorer {
public:
    using TYieldCtx = yplatform::yield_context<TStorer>;

    TStorer(TContextPtr ctx, TStoreRequest request, TCallback callback)
        : Ctx{std::move(ctx)}
        , Request{std::move(request)}
        , Callback{std::move(callback)}
    {
        Ctx->SetEnvelopeId(Request.Envelope.EnvelopeId);
    }

    void operator()(TYieldCtx yctx, TErrorCode ec = {}, TMessagePtr = {}) {
        try {
            reenter(yctx) {
                FillEnvelope();
                FillUsers();

                yield {
                    auto server = yplatform::find<IServer>("nsls");
                    server->Deliver(
                        Ctx,
                        Request.Message.Stid,
                        Envelope,
                        UserStorage,
                        NTimeTraits::Now(),
                        [&](const auto& message){UpdateMessageFromRequest(message, Request);},
                        [this](auto ctx, const auto& dataProvider, auto callback){
                            ProcessNewEmails(std::move(ctx), dataProvider, std::move(callback));},
                        yctx);
                }

                if (ec) {
                    Result = ec;
                    yield break;
                }

                FillResponse();
            }
        } catch (const std::exception& e) {
            NSLS_LOG_CTX_ERROR(
                logdog::message="Unexpected exception",
                logdog::exception=e,
                logdog::where_name=Where);
            return Callback(EError::DeliveryInternal, R"({"error":"internal delivery error"})");
        }

        if (yctx.is_complete()) {
            return Callback(Result, Response);
        }
    }

private:
    void FillEnvelope() {
        Envelope.Lhlo = Request.Envelope.Helo;
        Envelope.RemoteIp = Request.Envelope.RemoteIp;
        Envelope.RemoteHost = Request.Envelope.RemoteHost;
        Envelope.MailFrom = Request.Envelope.MailFrom.Email;
        Envelope.Hostname = Request.Envelope.RemoteHost;
    }

    void FillUsers() {
        UserStorage = std::make_shared<NUser::TStorage>();
        for (const auto& recipient : Request.Recipients) {
            UserStorage->AddUser(recipient.Email, true, true);
        }

        if (Request.Recipients.size() == 1) {
            const auto& recipient{Request.Recipients[0]};
            if (recipient.IsMailish.value_or(false)) {
                if (!recipient.Uid || recipient.Uid->empty()) {
                    throw std::logic_error("Empty uid for the mailish user");
                }

                UserStorage->SetMailish(*recipient.Uid);
            }
        } else if (AnyOf(Request.Recipients, [](const auto& recipient){return recipient.IsMailish.value_or(
            false);}))
        {
            throw std::logic_error("There can be one and only one user in a request with mailish users");
        }
    }

    void ProcessNewEmails(TContextPtr ctx, const NNewEmails::TDataProvider& dataProvider,
        NNewEmails::TProcessor::TSenderCallback callback)
    {
        yplatform::spawn(NDetail::TNewEmailsProcessor{std::move(ctx), dataProvider, NewRecipients,
            std::move(callback)});
    }

    void FillResponse() {
        TStoreResponse response;
        for (auto& requestRecipient : Request.Recipients) {
            TStoreResponseRecipient responseRecipient;
            responseRecipient.Email = std::move(requestRecipient.Email);
            responseRecipient.IsLocal = std::move(requestRecipient.IsLocal);
            responseRecipient.Notify = requestRecipient.Notify;

            const auto user{UserStorage->GetUserByEmail(responseRecipient.Email)};
            if (!user || user->Status != NUser::ELoadStatus::Found) {
                responseRecipient.Uid = std::move(requestRecipient.Uid);
                responseRecipient.Status = "perm_error";
            } else if (user->DeliveryResult.IsDuplicate) {
                responseRecipient.DeliveryResult = {TDeliveryResult{.Mid = user->DeliveryResult.Mid}};
                responseRecipient.Status = "deduplicated";
            } else if (user->DeliveryResult.ErrorCode == EError::DeliveryLoopDetected) {
                responseRecipient.Status = "decycled";
            } else if (NError::IsOk(user->DeliveryResult.ErrorCode)) {
                responseRecipient.DeliveryResult = {TDeliveryResult{.Mid = user->DeliveryResult.Mid,
                    .ImapId = user->DeliveryResult.ImapId}};
                responseRecipient.Status = "stored";
            } else if (NError::IsPermError(user->DeliveryResult.ErrorCode)) {
                responseRecipient.Status = "perm_error";
            } else {
                responseRecipient.Status = "temp_error";
            }

            response.Recipients.emplace_back(std::move(responseRecipient));
        }

        for (auto& newRecipient : NewRecipients) {
            TStoreResponseRecipient responseRecipient;
            responseRecipient.Email = std::move(newRecipient.Email);
            responseRecipient.Uid = std::move(newRecipient.Uid);
            responseRecipient.IsLocal = std::move(newRecipient.IsLocal);
            responseRecipient.Notify.Success = newRecipient.Notifies.Success;
            responseRecipient.Notify.Delay = newRecipient.Notifies.Delay;
            responseRecipient.Notify.Failure = newRecipient.Notifies.Failure;
            responseRecipient.DeliveryResult = {TDeliveryResult{.Stid = std::move(newRecipient.Stid)}};
            responseRecipient.Status = "new";

            response.Recipients.emplace_back(std::move(responseRecipient));
        }

        Result = EError::Ok;
        Response = yamail::data::serialization::toJson(response).str();
    }

private:
    TContextPtr Ctx;
    TStoreRequest Request;
    TCallback Callback;

    NNotSoLiteSrv::TEnvelope Envelope;
    NUser::TStoragePtr UserStorage;
    TNewRecipients NewRecipients;

    TErrorCode Result;
    std::string Response;
    const std::string Where {"HTTP"};
};

#include <yplatform/unyield.h>

} // namespace NDetail

void UpdateMessageFromRequest(TMessagePtr message, const TStoreRequest& request) {
    for (const auto& hint : request.Message.Hints) {
        message->AddParsedXYHint(NReflection::ParseHttpHint(hint));
    }

    message->CombineXYHints();
    message->SetStid(request.Message.Stid);

    using namespace NTimeTraits;
    message->SetStartMark(TSystemTimePoint{duration_cast<TSystemDuration>(TMilliSeconds{
        request.Message.Timemark})});
}

void Store(TContextPtr ctx, TStoreRequest request, TCallback callback) {
    auto storer = std::make_shared<NDetail::TStorer>(std::move(ctx), std::move(request), std::move(callback));
    yplatform::spawn(storer);
}

}
