#include "store.h"

#include <mail/notsolitesrv/src/http/handlers/impl/dryrun/dryrun.h>
#include <mail/notsolitesrv/src/http/handlers/impl/store/store.h>
#include <mail/notsolitesrv/src/http/types/reflection/request.h>
#include <mail/notsolitesrv/src/tskv/log.h>

#include <yamail/data/deserialization/yajl.h>

#include <string>

namespace NNotSoLiteSrv::NHttp {

namespace NDetail {

ECode MapErrorCodeToHttpCode(TErrorCode errc) {
    if (!errc) {
        return ECode::ok;
    } else if (errc == EError::HttpRequestParseError) {
        return ECode::bad_request;
    } else if (errc == EError::NotImplemented) {
        return ECode::not_implemented;
    } else if (NError::IsPermError(errc)) {
        return ECode::not_acceptable;
    } else {
        return ECode::internal_server_error;
    }
}

} // namespace NDetail

void TStore::OnComplete(THttpStreamPtr stream, TErrorCode errc, std::string response, TCallback callback) {
    auto code = NDetail::MapErrorCodeToHttpCode(errc);
    stream->set_code(code);
    stream->set_content_type("application", "json");
    stream->result_body(response);

    callback();
}

void TStore::OnError(TContextPtr ctx, THttpStreamPtr stream, TErrorCode errc, std::string reason,
    TCallback callback)
{
    NSLS_LOG_ERROR(ctx, logdog::message="Error while processing /store request: " + reason);
    std::string response{R"({"error":")" + reason + R"("})"};
    return OnComplete(stream, std::move(errc), std::move(response), std::move(callback));
}

void TStore::Execute(TContextPtr ctx, THttpStreamPtr stream, TCallback callback) {
    const std::string body{stream->request()->raw_body.begin(), stream->request()->raw_body.end()};
    TStoreRequest request;
    try {
        request = yamail::data::deserialization::fromJson<TStoreRequest>(body);
    } catch (const std::exception& ex) {
        return OnError(ctx, stream, EError::HttpRequestParseError, ex.what(), std::move(callback));
    }

    try {
        auto onComplete = [this, stream, callback](TErrorCode errc, std::string response) {
            OnComplete(stream, std::move(errc), std::move(response), callback);
        };

        if (stream->request()->url.param_value_cast<bool>("dry_run", false)) {
            NImpl::DryRun(std::move(request), onComplete);
        } else {
            NImpl::Store(ctx, std::move(request), onComplete);
        }
    } catch (const std::exception& ex) {
        return OnError(ctx, stream, EError::DeliveryInternal, ex.what(), std::move(callback));
    }
}

} // namespace NNotSoLiteSrv::NHttp
