#include "client.h"

#include <mail/notsolitesrv/src/config/mdbsave.h>
#include <mail/notsolitesrv/src/mdbsave/types/reflection/request.h>
#include <mail/notsolitesrv/src/mdbsave/types/reflection/response.h>
#include <mail/notsolitesrv/src/http/util.h>
#include <mail/notsolitesrv/src/tskv/log.h>

#include <yamail/data/deserialization/yajl.h>
#include <yamail/data/serialization/yajl.h>
#include <ymod_httpclient/url_encode.h>

#include <boost/asio.hpp>

namespace NNotSoLiteSrv::NMdbSave {

const std::string MDBSAVE{"MDBSAVE"};

TMdbSaveClient::TMdbSaveClient(TContextPtr ctx, TClusterCallPtr clusterCall, TTvmModulePtr tvmModule)
    : Ctx{std::move(ctx)}
    , ClusterCall{std::move(clusterCall)}
    , TvmModule{std::move(tvmModule)}
{
}

void TMdbSaveClient::MdbSave(boost::asio::io_context& ioContext, std::string sessionId,
    TMdbSaveRequest mdbsaveRequest, TMdbSaveCallback callback) const
{
    auto clusterCallCallback = [=, executor = ioContext.get_executor()](auto&& errorCode, auto&& response) {
        boost::asio::post(executor, [this, callback, errorCode, response=std::move(response)]{
            ProcessMdbSaveResponse(
                std::move(errorCode),
                std::move(response),
                std::move(callback)
            );
        });
    };

    try {
        ClusterCall->async_run(
            Ctx->GetTaskContext(MDBSAVE),
            MakeMdbSaveRequest(std::move(sessionId), mdbsaveRequest),
            std::move(clusterCallCallback)
        );
    } catch (EError error) {
        auto errorCode = make_error_code(error);
        NSLS_LOG_CTX_ERROR(log::error_code = errorCode, log::where_name = MDBSAVE);
        boost::asio::post(ioContext, [=]{callback(std::move(errorCode), {});});
    }
}

THttpRequest TMdbSaveClient::MakeMdbSaveRequest(std::string sessionId,
    const TMdbSaveRequest& mdbsaveRequest) const
{
    auto url = "/1/save" + yhttp::url_encode({{"service", "nsls"}, {"session_id", sessionId}});
    if (Ctx->GetConfig()->MdbSave->UseTvm) {
        const auto ticket = GetTvmServiceTicket();
        return THttpRequest::POST(std::move(url), MakeTvmServiceTicketHeader(ticket),
            yamail::data::serialization::toJson(mdbsaveRequest));
    } else {
        return THttpRequest::POST(std::move(url), yamail::data::serialization::toJson(mdbsaveRequest));
    }
}

std::string TMdbSaveClient::GetTvmServiceTicket() const {
    std::string ticket;
    const auto errorCode = TvmModule->get_service_ticket(Ctx->GetConfig()->MdbSave->Service, ticket);
    if (errorCode) {
        NSLS_LOG_CTX_ERROR(log::error_code = errorCode, log::where_name = MDBSAVE);
        throw EError::TvmServiceTicketError;
    }

    return ticket;
}

void TMdbSaveClient::ProcessMdbSaveResponse(TErrorCode errorCode, THttpResponse response,
    TMdbSaveCallback callback) const
{
    if (errorCode || (response.status != 200)) {
        ProcessError(std::move(errorCode), std::move(response), std::move(callback));
    } else {
        auto mdbsaveResult = ParseMdbSaveResponse(std::move(response.body));
        if (!mdbsaveResult) {
            errorCode = make_error_code(EError::MdbSaveResponseParseError);
        }

        callback(std::move(errorCode), std::move(mdbsaveResult));
    }
}

TMdbSaveResult TMdbSaveClient::ParseMdbSaveResponse(std::string responseBody) const
{
    try {
        return yamail::data::deserialization::fromJson<TMdbSaveResponse>(responseBody);
    } catch (const std::exception& ex) {
        NSLS_LOG_CTX_ERROR(log::message = "failed to parse mdbsave response", log::exception = ex,
            log::where_name = MDBSAVE);
        return {};
    }
}

void TMdbSaveClient::ProcessError(TErrorCode errorCode, THttpResponse response,
    TMdbSaveCallback callback) const
{
    if (errorCode) {
        NSLS_LOG_CTX_ERROR(log::error_code = errorCode, log::where_name = MDBSAVE);
    } else if ((response.status / 100) != 5) {
        errorCode = make_error_code(EError::HttpNonRetryableStatus);
        NSLS_LOG_CTX_ERROR(log::message = MakeErrorStatusMessage(std::move(response.body)),
            log::error_code = errorCode, log::where_name = MDBSAVE);
    } else {
        errorCode = make_error_code(EError::HttpRetriesExceeded);
        NSLS_LOG_CTX_ERROR(log::message = MakeErrorStatusMessage(std::move(response.body)),
            log::error_code = errorCode, log::where_name = MDBSAVE);
    }

    callback(std::move(errorCode), {});
}

std::string TMdbSaveClient::MakeErrorStatusMessage(std::string responseBody) const {
    auto errorResult = ParseErrorResponse(std::move(responseBody));
    if (!errorResult) {
        return {};
    }

    return errorResult->Error + " (" + errorResult->Message + ")";
}

TMdbSaveClient::TErrorResult TMdbSaveClient::ParseErrorResponse(std::string responseBody) const {
    try {
        return yamail::data::deserialization::fromJson<TErrorResponse>(responseBody);
    } catch (const std::exception& ex) {
        NSLS_LOG_CTX_ERROR(log::message = "failed to parse error response", log::exception = ex,
            log::where_name = MDBSAVE);
        return {};
    }
}

}
