#include "message.h"
#include <mail/notsolitesrv/src/reflection/xyandexhint.h>
#include <mail/notsolitesrv/src/blackwhitelist/list.h>
#include <mail/notsolitesrv/src/config/message.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/notsolitesrv/src/util/headers.h>
#include <mail/notsolitesrv/src/util/log.h>
#include <mail/notsolitesrv/src/util/message.h>
#include <mail/notsolitesrv/src/util/string.h>
#include <mail/notsolitesrv/src/util/zerocopy.h>
#include <mail/library/utf8/utf8.h>
#include <mimeparser/rfc2822date.h>
#include <yplatform/zerocopy/streambuf.h>
#include <boost/algorithm/string/join.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/cxx11/copy_if.hpp>
#include <boost/algorithm/cxx11/none_of.hpp>
#include <boost/range/algorithm_ext/erase.hpp>
#include <boost/tokenizer.hpp>
#include <library/cpp/digest/md5/md5.h>
#include <util/generic/algorithm.h>
#include <util/generic/is_in.h>
#include <util/generic/yexception.h>

#include <sstream>
#include <utility>

namespace NNotSoLiteSrv {

ESpamType SpamTypeFromXYSpam(int xySpam) {
    switch (xySpam) {
        case 1:
            return ESpamType::Ham;
        case 2:
            return ESpamType::Delivery;
        case 4:
            return ESpamType::Spam;
        case 256:
            return ESpamType::Malicious;
        default:
            return ESpamType::Unknown;
    }
}

bool IsSpam(ESpamType spamType) {
    switch (spamType) {
        case ESpamType::Ham:
        case ESpamType::Delivery:
        case ESpamType::Unknown:
            return false;

        case ESpamType::Spam:
        case ESpamType::Malicious:
            return true;
    }
}

TAttachment::TAttachment(const TPart& part, std::string fname) {
    Type = NUtil::StripBadChars(::NUtil::Utf8Sanitized(part.GetContentType())) + "/"
        + NUtil::StripBadChars(::NUtil::Utf8Sanitized(part.GetContentSubtype()));
    Size = NUtil::CalculateAttachmentSize(part.GetEncoding(), part.GetBody());

    if (fname.empty()) {
        fname = part.GetFilename().empty() ? "Noname" : part.GetFilename();
    }
    Name = NUtil::StripBadChars(::NUtil::Utf8Sanitized(fname));
}

TMessage::TMessage(TPart&& rootPart, TContextPtr ctx, bool isHttpSession)
    : TPart(std::move(rootPart))
    , Ctx(ctx)
    , ReceivedDate(time(nullptr))
    , IsHttpSession(isHttpSession)
{
    Body.append(NUtil::MakeZerocopySegment(GetBody()));
}

TMessagePtr TMessage::CreateMessageFromRootPart(TPart&& rootPart, TContextPtr ctx, bool isHttpSession) {
    auto msgPtr = new TMessage(std::move(rootPart), ctx, isHttpSession);
    TMessagePtr msg;
    try {
        msg = TMessagePtr(msgPtr);
    } catch (...) {
        delete msgPtr;
        throw;
    }
    msg->SetNewParent(msg);
    if (auto partsCount = std::distance(msg->begin(), msg->end()); partsCount > ctx->GetConfig()->Message->MaxParts) {
        throw std::runtime_error("Too many parts (" + std::to_string(partsCount) +
            ") while allowed only " + std::to_string(ctx->GetConfig()->Message->MaxParts));
    }

    try {
        msg->FindAttachments();
    } catch (const std::exception&) {
        // ignore
    }
    return msg;
}

constexpr size_t HEADER_BODY_DELIM_LEN = 2; // CRLF

off_t TMessage::GetOffsetDiff() const {
    if (IsHttpSession) {
        return 0;
    }
    size_t size;
    auto hint = GetXYHint();
    if (IsMeta() && hint && hint->final_headers_len) {
        size = *hint->final_headers_len;
    } else {
        size = FinalHeaders.size();
    }
    Y_ENSURE(size < Max<off_t>() - HEADER_BODY_DELIM_LEN);
    return GetOffset() - size - HEADER_BODY_DELIM_LEN;
}

size_t TMessage::GetPersonalHeadersLength(const std::string& uid) const {
    if (IsHttpSession) {
        return 0;
    }
    auto addedHeadersIter = HeadersForUid.find(uid);
    if (addedHeadersIter == HeadersForUid.end()) {
        return 0;
    }

    return addedHeadersIter->second.size();
}

size_t TMessage::GetFinalLength() const {
    if (IsHttpSession) {
        return GetOffset() + GetLength();
    }
    Y_ENSURE(GetLength() < Max<size_t>() - HEADER_BODY_DELIM_LEN);
    size_t size;
    auto hint = GetXYHint();
    if (IsMeta() && hint && hint->final_headers_len) {
        size = *hint->final_headers_len;
    } else {
        size = FinalHeaders.size();
    }
    Y_ENSURE(size < Max<size_t>() - GetLength() - HEADER_BODY_DELIM_LEN);
    return size + GetLength() + HEADER_BODY_DELIM_LEN;
}

bool TMessage::IsPartApplicableForDB(const TPart& part) const {
    if (!Ctx->GetConfig()->Message->DoNotStoreMessageRfc822SubpartsHack) {
        return true;
    }

    // Skip message/rfc822 sub-parts due to buggy IMAP
    bool skip = false;
    for (auto parent = part.GetParent().lock(); parent; parent = parent->GetParent().lock()) {
        if (boost::iequals(parent->GetContentType(), "message") &&
            boost::iequals(parent->GetContentSubtype(), "rfc822"))
        {
            skip = true;
            break;
        }
    }

    return !skip;
}

void TMessage::FindAttachments() {
    using namespace NUtil;
    auto cfg = Ctx->GetConfig();
    for (const auto& part: *this) {
        if (part.GetContentType().empty()) {
            continue;
        }

        // Skip multipart
        if (boost::iequals(part.GetContentType(), "multipart")) {
            continue;
        }

        std::string fname = part.GetFilename().empty() ? part.GetName() : part.GetFilename();

        // Skip inline attachments, but not YaDisk (MPROTO-9, MPROTO-2462)
        if (boost::iequals(part.GetContentDisposition(), "inline") &&
            !boost::iequals(fname, "narod_attachment_links.html"))
        {
            continue;
        }

        // Skip PKCS7 signature attachments
        if (boost::iequals(fname, "smime.p7s") ||
            (boost::iequals(part.GetContentType(), "application") &&
             boost::iequals(part.GetContentSubtype(), "pkcs7-signature")))
        {
            continue;
        }

        // Skip PGP signature attachments
        if (boost::iequals(part.GetContentType(), "application") &&
            boost::iequals(part.GetContentSubtype(), "pgp-signature"))
        {
            continue;
        }

        // Skip parts with Content-Id header
        if (!part.GetContentId().empty() && cfg->Message->SkipAttachWithContentId) {
            continue;
        }

        // Skip message/rfc822 sub-parts due to DB constraints
        bool skip = false;
        for (auto parent = part.GetParent().lock(); parent; parent = parent->GetParent().lock()) {
            if (boost::iequals(parent->GetContentType(), "message") &&
                boost::iequals(parent->GetContentSubtype(), "rfc822"))
            {
                skip = true;
                break;
            }
        }
        if (skip) {
            continue;
        }

        // Skip text sub-part of multipart/alternative
        auto parent = part.GetParent().lock();
        if (parent &&
            boost::iequals(parent->GetContentType(), "multipart") &&
            boost::iequals(parent->GetContentSubtype(), "alternative"))
        {
            if (boost::iequals(part.GetContentType(), "text")) {
                continue;
            } else if (fname.empty()) {
                fname = "Noname";
            }
        }

        // Yahoo! Save attachment

        // Special case for message/rfc822 parts: use first sub-part's subject as a filename
        if (boost::iequals(part.GetContentType(), "message") &&
            boost::iequals(part.GetContentSubtype(), "rfc822"))
        {
            std::string subject;
            if (fname.empty() && !part.GetChildren().empty()) {
                for (const auto& hdr: part.GetChildren().front()->GetHeaders()) {
                    if (boost::iequals(hdr.first, "subject")) {
                        subject = StripBadChars(::NUtil::Utf8Sanitized(DecodeHeaderRfc2047(hdr.second)));
                        subject = ::NUtil::Utf8ByteHead(subject, NConstants::MAX_FNAME_LEN - 4);
                        break;
                    }
                }
            }

            if (fname.empty()) {
                if (subject.empty()) {
                    fname = "Noname.eml";
                } else {
                    fname = subject + ".eml";
                }
            }
        } else {
            // Skip parts without name or filename
            if (fname.empty()) {
                continue;
            }
            fname = StripBadChars(::NUtil::Utf8Sanitized(DecodeHeaderRfc2047(fname)));
            fname = FilenameSanitized(fname, NConstants::MAX_FNAME_LEN);
        }

        Attachments.emplace(part.GetHid(), TAttachment{part, fname});
    }
}

void TMessage::AddHeader(const std::string& name, const std::string& value) {
    FinalHeaders.append(NUtil::MakeZerocopySegment(name, ": ", value, "\r\n"));
}

bool TMessage::HasPersonalHeaders(const std::string& uid) const {
    return HeadersForUid.find(uid) != HeadersForUid.end();
}

yplatform::zerocopy::segment TMessage::Compose() const {
    yplatform::zerocopy::segment ret = FinalHeaders;
    ret.append(NUtil::MakeZerocopySegment("\r\n")).append(Body);

    return ret;
}

yplatform::zerocopy::segment TMessage::Compose(const std::string& uid) const {
    yplatform::zerocopy::segment ret = FinalHeaders;
    if (auto addedHeadersIter = HeadersForUid.find(uid); addedHeadersIter != HeadersForUid.end()) {
        ret.append(addedHeadersIter->second);
    }
    ret.append(NUtil::MakeZerocopySegment("\r\n")).append(Body);

    return ret;
}

std::string TMessage::GetStid(const std::string& uid) const {
    if (auto personalStid = GetPersonalStid(uid)) {
        return *personalStid;
    }

    return Stid;
}

boost::optional<std::string> TMessage::GetPersonalStid(const std::string& uid) const {
    if (auto stidForUidIter = StidForUid.find(uid); stidForUidIter != StidForUid.end()) {
        return stidForUidIter->second;
    }

    return {};
}

void TMessage::SetSpamTypeFromXYSpam(int xySpam) {
    SpamType = SpamTypeFromXYSpam(xySpam);
}

bool TMessage::IsSpam() const {
    return NNotSoLiteSrv::IsSpam(SpamType);
}

void TMessage::SetSpamTypeByUid(const std::string& uid, ESpamType spamType) {
    SpamTypeByUid[uid] = spamType;
}

ESpamType TMessage::GetSpamType(const std::string& uid) const {
    if (SpamTypeByUid.contains(uid)) {
        return SpamTypeByUid.at(uid);
    } else {
        return GetSpamType();
    }
}

bool TMessage::IsSpam(const std::string& uid) const {
    if (SpamTypeByUid.contains(uid)) {
        return NNotSoLiteSrv::IsSpam(SpamTypeByUid.at(uid));
    } else {
        return IsSpam();
    }
}

ESpamResolutionType TMessage::GetSpamResolutionType(const std::string& uid) const {
    if (SpamTypeByUid.contains(uid)) {
        return ESpamResolutionType::PersonalByUid;
    } else {
        return ESpamResolutionType::Global;
    }
}

constexpr int MAX_ALLOWED_YEAR = 9999;

void TMessage::SetDate(const std::string& val) {
    rfc2822::rfc2822date dt(val);
    time_t datetime = dt.unixtime();
    if (datetime < 0) {
        return;
    }

    struct tm tinfo;
    localtime_r(&datetime, &tinfo);
    if (tinfo.tm_year + 1900 <= MAX_ALLOWED_YEAR) {
        Date = datetime;
    }
}

void TMessage::SetPrecedence(const std::string& val) {
    Precedence = boost::trim_copy(::NUtil::Utf8Sanitized(val));
}

void TMessage::SetAutoSubmitted(const std::string& val) {
    const auto trimmedValue = boost::trim_copy(val);
    AutoSubmitted = (trimmedValue == "auto-generated" || trimmedValue == "auto-replied");
}

void TMessage::SetSubject(const std::string& val) {
    Subject = NUtil::StripBadChars(::NUtil::Utf8Sanitized(NUtil::DecodeHeaderRfc2047(val, GetCharset())));
}

void TMessage::SetMessageId(const std::string& val) {
    auto rawMessageId = NUtil::StripBadChars(::NUtil::Utf8Sanitized(val));
    if (!NUtil::ParseMessageId(rawMessageId, MessageId)) {
        MessageId = rawMessageId;
    }
}

void TMessage::SetReferences(const std::string& val) {
    std::vector<std::string> references;
    boost::split(
        references,
        val,
        boost::is_any_of(" \t\r\n"),
        boost::token_compress_on);

    if (references.empty()) {
        return;
    }

    for (const auto& ref: references) {
        std::string msgId;
        if (NUtil::ParseMessageId(::NUtil::Utf8Sanitized(ref), msgId)) {
            References.emplace_back(std::move(msgId));
        }
    }
}

void TMessage::SetInReplyTo(const std::string& val) {
    if (!NUtil::ParseMessageId(::NUtil::Utf8Sanitized(val), InReplyTo)) {
        InReplyTo.clear();
    }
}

void TMessage::SetAuthResults(const std::string& val) {
    AuthResults = ::NUtil::Utf8Sanitized(val);
}

void TMessage::SetSender(const std::string& val) {
    Sender = ::NUtil::Utf8Sanitized(val);
}

void TMessage::SetReplyTo(const std::string& val) {
    rfc2822::old_rfc2822address addrList(val);
    for (const auto& addr: addrList) {
        if (!ReplyTo.empty()) {
            ReplyTo.append(",");
        }
        ReplyTo.append(::NUtil::Utf8Sanitized(addr.second));
    }
}

std::string TMessage::GetReplyTo() const {
    if (!ReplyTo.empty()) {
        return ReplyTo;
    }

    std::string ret;
    if (GetFrom() && !GetFrom()->empty()) {
        for (const auto& addr: *GetFrom()) {
            if (!ret.empty()) {
                ret.append(",");
            }
            ret.append(addr.second);
        }
    }

    if (ret.empty() && !Sender.empty()) {
        rfc2822::old_rfc2822address addrList(Sender);
        for (const auto& addr: addrList) {
            if (!ret.empty()) {
                ret.append(",");
            }
            ret.append(addr.second);
        }
    }

    return ::NUtil::Utf8Sanitized(ret);
}

void TMessage::SetAutoReply(const std::string& val) {
    if (!val.empty()) {
        AutoReply = true;
    }
}

void TMessage::AddParsedXYHint(TXYandexHint&& hint) {
    Hints.emplace_back(hint);
}

void TMessage::AddXYHint(const std::string& val) {
    AddParsedXYHint(NReflection::ParseXYHint(val));
}

void TMessage::CombineXYHints() {
    StableSortBy(Hints, [](const auto& hint) { return hint.email; });
    boost::optional<std::string> lastEmail;
    decltype(Hints) tmpHints;
    for (const auto& hint: Hints) {
        if (!lastEmail || *lastEmail != hint.email) {
            lastEmail = hint.email;
            tmpHints.emplace_back(std::move(hint));
            continue;
        }

        NReflection::CombineXYHint(tmpHints.back(), hint);
    }

    Hints.swap(tmpHints);

    if (!Hints.empty()) {
        if (Hints.front().received_date != 0) {
            ReceivedDate = Hints.front().received_date;
        }
        if (Hints.front().hdr_date != 0) {
            Date = Hints.front().hdr_date;
        }
    }
}

void TMessage::ResolveXYHints(const NUser::TStorage& userStorage) {
    for (auto& hint: Hints) {
        NSLS_LOG_CTX_DEBUG(logdog::message="X-Yandex-Hint: " + hint.ToString());
        std::string uid;
        if (!hint.email.empty()) {
            const auto& user = userStorage.GetUserByEmail(hint.email);
            if (!user || user->Status != NUser::ELoadStatus::Found || user->Uid.empty()) {
                NSLS_LOG_CTX_WARN(logdog::message="skip personal hint for " + hint.email + ": user not found");
                continue;
            }
            uid = user->Uid;
        }
        auto inserted = HintByUid.emplace(uid, hint);
        if (!inserted.second) {
            NReflection::CombineXYHint(inserted.first->second, hint);
        }
    }
}

void TMessage::SanitizeXYHints() {
    for (auto& [uid, hint]: HintByUid) {
        if (uid.empty()) {
            // params that can be only in targetted XYHint
            hint.copy_to_inbox = 0;
            hint.save_to_sent = boost::none;
        } else {
            if (hint.save_to_sent != boost::none) {
                if (hint.save_to_sent == false && hint.copy_to_inbox) {
                    // if we are going to save email as deleted in \Sent and copy it to \Inbox -
                    // store it to \Inbox only
                    hint.save_to_sent = boost::none;
                    hint.copy_to_inbox = false;
                    hint.filters = true;
                    hint.notify = true;
                } else {
                    // do not apply filters for \Sent messages
                    hint.filters = false;
                    hint.notify = false;
                }
            }
            // reset params that MUST NOT exists in targetted XYHint
            hint.source_stid.clear();
            hint.body_md5.clear();
        }
    }
}

boost::optional<const TXYandexHint&> TMessage::GetXYHintByEmail(const std::string& email) const {
    for (const auto& hint: Hints) {
        if (boost::iequals(hint.email, email)) {
            return hint;
        }
    }

    return boost::none;
}

boost::optional<const TXYandexHint&> TMessage::GetXYHint() const {
    if (auto hintIter = HintByUid.find(""); hintIter != HintByUid.end()) {
        return hintIter->second;
    }

    return {};
}

TXYandexHint TMessage::GetXYHintByUid(uint64_t uid) const {
    return GetXYHintByUid(std::to_string(uid));
}

TXYandexHint TMessage::GetXYHintByUid(const std::string& uid) const {
    Y_ENSURE(!uid.empty());

    TXYandexHint hint;

    if (HintByUid.empty()) {
        return hint;
    }

    auto hintIter = HintByUid.find(uid);
    if (hintIter != HintByUid.end()) {
        hint = hintIter->second;
    }

    if (auto defaultHint = GetXYHint()) {
        if (!hint.replace_so_labels) {
            NReflection::CombineXYHint(hint, *defaultHint);
        } else {
            std::vector<std::string> soLabels;
            boost::algorithm::copy_if(hint.label, std::back_inserter(soLabels), [](const auto& l) {
                return boost::algorithm::starts_with(l, "SystMetkaSO:");
            });

            NReflection::CombineXYHint(hint, *defaultHint);

            boost::range::remove_erase_if(hint.label, [&](const auto& l) {
                return boost::algorithm::starts_with(l, "SystMetkaSO:")
                    && boost::algorithm::none_of(soLabels, [&](const auto& s) { return l == s; });
            });
        }
    }

    return hint;
}

void TMessage::AddXYForwardValue(const std::string& val) {
    XYForwardValues.emplace(::NUtil::Utf8Sanitized(val));
}

bool TMessage::IsInXYForwards(const std::string& suid) const {
    return IsIn(XYForwardValues, NUtil::MakeXYForwardValue(suid));
}

void TMessage::AddSpamTypeByUid(const std::string& val) {
    decltype(SpamTypeByUid) spamTypeByUid;
    boost::char_separator<char> separator(",");
    boost::tokenizer<decltype(separator)> tokens(val, separator);
    for (const auto& token: tokens) {
        const int unknownSpamTypeCode{-1};
        int spamTypeCode{unknownSpamTypeCode};
        std::string uid;
        std::istringstream(token) >> spamTypeCode >> uid;
        if ((spamTypeCode == unknownSpamTypeCode) || uid.empty()) {
            return;
        }

        spamTypeByUid.emplace(uid, SpamTypeFromXYSpam(spamTypeCode));
    }

    SpamTypeByUid = std::move(spamTypeByUid);
}

void TMessage::StoreAsDeleted(const std::string& uid) {
    auto hintIter{HintByUid.find(uid)};
    if (hintIter == HintByUid.end()) {
        TXYandexHint newHint;
        newHint.folder_path = "\\inbox";
        newHint.folder_spam_path = "\\inbox";
        newHint.filters = false;
        newHint.store_as_deleted = true;
        Hints.push_back(newHint);
        HintByUid.emplace(uid, Hints.back());
    } else {
        hintIter->second.save_to_sent = boost::none;
        hintIter->second.copy_to_inbox = false;
        hintIter->second.fid = "";
        hintIter->second.folder = "";
        hintIter->second.folder_path = "\\inbox";
        hintIter->second.folder_spam_path = "\\inbox";
        hintIter->second.filters = false;
        hintIter->second.store_as_deleted = true;
    }
}

TErrorCode TMessage::UpdateResolvedUsers(NUser::TStorage& userStorage) {
    ResolveXYHints(userStorage);
    SanitizeXYHints();
    UpdateMetaFlagAndStid(Ctx);

    auto&& filter = [](const auto& user) {
        return user.Status == NUser::ELoadStatus::Found && user.DeliveryParams.NeedDelivery;
    };
    for (auto& [login, user]: userStorage.GetFilteredUsers(filter)) {
        const auto hint{GetXYHintByUid(user.Uid)};
        if (userStorage.IsMailish()) {
            if (hint.external_imap_id == 0) {
                return EError::DeliveryCorruptedParams;
            }
            return EError::Ok;
        }

        if (user.OrgId) {
            user.DeliveryParams.UseDomainRules = hint.filters;
        }

        if (user.DeliveryParams.BWList && GetFrom() && hint.filters) {
            bool bwlist_filtered = false;
            if (IsSpam(user.Uid)) {
                for (const auto& from: *GetFrom()) {
                    const auto& addr = from.second;
                    if (user.DeliveryParams.BWList->Check(NBlackWhiteList::EType::White, addr)) {
                        NSLS_LOG_CTX_DEBUG(logdog::message="WhiteList was applied for " + addr);
                        SetSpamTypeByUid(user.Uid, ESpamType::Ham);
                        bwlist_filtered = true;
                        break;
                    }
                }
            }
            if (!bwlist_filtered) {
                for (const auto& from: *GetFrom()) {
                    const auto& addr = from.second;
                    if (user.DeliveryParams.BWList->Check(NBlackWhiteList::EType::Black, addr)) {
                        NSLS_LOG_CTX_DEBUG(logdog::message="BlackList was applied for " + addr);
                        SetSpamTypeByUid(user.Uid, ESpamType::Spam);
                        StoreAsDeleted(user.Uid);
                        break;
                    }
                }
            }
        }

        if (!IsMeta() && !hint.bcc.empty()) {
            AddHeaderForUid(user.Uid, "BCC", boost::join(hint.bcc, ","));
        }

        if (hint.imap ||
            hint.skip_loop_prevention ||
            IsIn(hint.skip_loop_prevention_for_suid, user.Suid))
        {
            continue;
        }

        auto xyf = NUtil::MakeXYForwardValue(user.Suid);
        if (IsIn(XYForwardValues, xyf)) {
            user.DeliveryResult.ErrorCode = EError::DeliveryLoopDetected;
            continue;
        } else {
            AddHeader("X-Yandex-Forward", xyf);
        }
    }

    return EError::Ok;
}

void TMessage::AddHeaderForUid(const std::string& uid, const std::string& name, const std::string& value) {
    HeadersForUid[uid].append(NUtil::MakeZerocopySegment(name, ": ", value, "\r\n"));
}

bool TMessage::IsMetaNeeded() const {
    if (IsMeta()) {
        return false;
    }

    auto hint = GetXYHint();
    if (!hint) {
        return true;
    }
    if (hint->skip_meta_msg || hint->imap || hint->external_imap_id) {
        return false;
    }

    return true;
}

void TMessage::UpdateMetaFlagAndStid(TContextPtr ctx) {
    Meta = false;
    auto hint = GetXYHint();
    if (!hint) {
        return;
    }

    if (hint->source_stid.empty()) {
        return;
    }

    std::string skipMetaReason;
    if (hint->skip_meta_msg) {
        skipMetaReason = "skip_meta_msg=1";
    } else if (hint->imap) {
        skipMetaReason = "imap=1";
    } else if (hint->external_imap_id) {
        skipMetaReason = "mailish";
    } else if (hint->body_md5.empty()) {
        skipMetaReason = "empty body_md5";
    } else if (hint->final_headers_len == boost::none) {
        skipMetaReason = "empty final_headers_len";
    } else {
        std::string hintMd5 = MD5::Calc(GetBody());
        if (hintMd5 != hint->body_md5) {
            skipMetaReason = "body_md5 mismatch: got " + hint->body_md5 + ", expected " + hintMd5;
        }
    }

    if (!skipMetaReason.empty()) {
        NSLS_LOG_NOTICE(ctx, logdog::message="skip meta mail: " + skipMetaReason);
        NUtil::LogStidForRemove(ctx, hint->source_stid);
    } else {
        SetStid(hint->source_stid);
        Meta = true;
    }
}

} // namespace NNotSoLiteSrv
