#pragma once

#include "part.h"
#include "xyandexhint.h"
#include <mail/notsolitesrv/src/user/storage.h>
#include <mail/notsolitesrv/src/types/time.h>
#include <mail/notsolitesrv/src/util/message.h>
#include <mail/notsolitesrv/src/util/md5.h>
#include <mail/notsolitesrv/src/constants.h>
#include <mail/notsolitesrv/src/context.h>
#include <yplatform/log.h>
#include <yplatform/zerocopy/segment.h>
#include <boost/optional.hpp>
#include <map>
#include <set>
#include <vector>

namespace NNotSoLiteSrv {

enum class ESpamType {
    Unknown,
    Ham,
    Delivery,
    Spam,
    Malicious
};

enum class ESpamResolutionType {
    Global,
    PersonalByUid
};

struct TAttachment {
    explicit TAttachment(const TPart& part, std::string fname = "");
    std::string Type;
    std::string Name;
    uint64_t Size;
};

using TAttachments = std::map<THid, TAttachment>;

class TMessage: public TPart {
public:
    TMessage() = delete;

    static std::shared_ptr<TMessage> CreateMessageFromRootPart(TPart&& rootPart, TContextPtr ctx, bool isHttpSession);

    void AddHeader(const std::string& name, const std::string& value);
    yplatform::zerocopy::segment Compose() const;
    yplatform::zerocopy::segment Compose(const std::string& uid) const;

    off_t GetOffsetDiff() const;
    size_t GetPersonalHeadersLength(const std::string& uid) const;
    size_t GetFinalHeadersLength() const { return FinalHeaders.size(); }
    size_t GetFinalLength() const;
    bool HasPersonalHeaders(const std::string& uid) const;

    void StoreAsDeleted(const std::string& uid);
    TErrorCode UpdateResolvedUsers(NUser::TStorage& userStorage);

    void SetStid(const std::string& stid) { Stid = stid; }
    void SetStid(const std::string& stid, const std::string& uid) { StidForUid.emplace(uid, stid); }
    const std::string& GetStid() const { return Stid; }
    std::string GetStid(const std::string& uid) const;
    boost::optional<std::string> GetPersonalStid(const std::string& uid) const;

    void SetSpamType(ESpamType spamType) { SpamType = spamType; }
    void SetSpamTypeFromXYSpam(int xySpam);
    ESpamType GetSpamType() const { return SpamType; }
    bool IsSpam() const;

    void SetSpamTypeByUid(const std::string& uid, ESpamType spamType);
    ESpamType GetSpamType(const std::string& uid) const;
    bool IsSpam(const std::string& uid) const;

    ESpamResolutionType GetSpamResolutionType(const std::string& uid) const;

    time_t GetReceivedDate() const { return ReceivedDate; }

    void SetDate(const std::string& val);
    time_t GetDate() const { return Date; }

    void SetPrecedence(const std::string& val);
    std::string GetPrecedence() const { return Precedence; }

    void SetAutoSubmitted(const std::string& val);
    bool IsAutoSubmitted() const { return AutoSubmitted; }

    void SetSubject(const std::string& val);
    const std::string& GetSubject() const { return Subject; }

    void SetMessageId(const std::string& val);
    const std::string& GetMessageId() const { return MessageId; }

    void SetSender(const std::string& val);
    const std::string& GetSender() const { return Sender; }

    void SetReferences(const std::string& val);
    const auto& GetReferences() const { return References; }

    void SetInReplyTo(const std::string& val);
    const std::string& GetInReplyTo() const { return InReplyTo; }

    void SetAuthResults(const std::string& val);

    void SetReplyTo(const std::string& val);
    std::string GetReplyTo() const;

    void SetAutoReply(const std::string& val);
    bool IsAutoReply() const { return AutoReply; }

    void SetDSN(bool dsn) { Dsn = dsn; }
    bool IsDSN() const { return Dsn; }

    void AddXYHint(const std::string& val);
    void AddParsedXYHint(TXYandexHint&& hint);
    void CombineXYHints();
    boost::optional<const TXYandexHint&> GetXYHint() const;
    boost::optional<const TXYandexHint&> GetXYHintByEmail(const std::string& email) const;
    TXYandexHint GetXYHintByUid(uint64_t uid) const;
    TXYandexHint GetXYHintByUid(const std::string& uid) const;
    const auto& GetXYHints() const { return Hints; }

    void AddXYForwardValue(const std::string& val);
    bool IsInXYForwards(const std::string& suid) const;
    const auto& GetXYForwardValues() const { return XYForwardValues; }
    void AddSpamTypeByUid(const std::string& val);

    const TAttachments& GetAttachments() const { return Attachments; }
    bool IsMeta() const { return Meta; }
    bool IsMetaNeeded() const;

    void SetStartMark(const NTimeTraits::TSystemTimePoint& startMark) { StartMark = startMark; }
    const NTimeTraits::TSystemTimePoint& GetStartMark() const { return StartMark; }
    void SetDeliveryStartMark(const NTimeTraits::TSystemTimePoint& startMark) { DeliveryStartMark = startMark; }
    const NTimeTraits::TSystemTimePoint& GetDeliveryStartMark() const { return DeliveryStartMark; }

    bool IsPartApplicableForDB(const TPart& part) const;

private:
    TMessage(TPart&& rootPart, TContextPtr ctx, bool isHttpSession);

    void ResolveXYHints(const NUser::TStorage& userStorage);
    void SanitizeXYHints();
    void FindAttachments();
    void UpdateMetaFlagAndStid(TContextPtr ctx);
    void AddHeaderForUid(const std::string& uid, const std::string& name, const std::string& value);

private:
    TContextPtr Ctx;

    yplatform::zerocopy::segment FinalHeaders;
    yplatform::zerocopy::segment Body;
    ESpamType SpamType = ESpamType::Unknown;
    std::string Stid;

    time_t Date = 0;
    time_t ReceivedDate = 0;
    std::string Precedence;
    bool AutoReply = false;
    bool AutoSubmitted = false;
    bool Dsn = false;
    bool Meta = false;
    std::string Subject;
    std::string MessageId;
    std::vector<std::string> References;
    std::string InReplyTo;
    std::string AuthResults;
    std::string ReplyTo;
    std::string Sender;

    std::vector<TXYandexHint> Hints;
    std::map<std::string, TXYandexHint> HintByUid;
    std::set<std::string> XYForwardValues;
    std::map<std::string, ESpamType> SpamTypeByUid;
    TAttachments Attachments;
    std::map<std::string, yplatform::zerocopy::segment> HeadersForUid;
    std::map<std::string, std::string> StidForUid;

    NTimeTraits::TSystemTimePoint StartMark = NTimeTraits::FromUnixTime(0); // timestamp when Y.Mail (nwsmtp) receives message
    NTimeTraits::TSystemTimePoint DeliveryStartMark = NTimeTraits::Now(); // timestamp when NSLS starts processing

    bool IsHttpSession = false;
};
using TMessagePtr = std::shared_ptr<TMessage>;

} // namespace NNotSoLiteSrv
