#include "message_handler.h"
#include "rfc2231.h"
#include <mail/notsolitesrv/src/message/parser.h>
#include <mail/notsolitesrv/src/util/headers.h>
#include <mail/notsolitesrv/src/util/recognizer.h>
#include <mail/library/utf8/utf8.h>
#include <mail_getter/UTFizer.h>
#include <mimeparser/ccnv.h>
#include <mimeparser/rfc2047.h>
#include <boost/algorithm/string/trim.hpp>

namespace NNotSoLiteSrv {

template <typename TIterator, typename THHandler>
bool TMessageHandler<TIterator, THHandler>::beginMessage(const TIterator& pos) {
    Start = pos;
    return true;
}

template <typename TIterator, typename THHandler>
bool TMessageHandler<TIterator, THHandler>::beginPart(const TIterator& pos, const THHandler& hHandler) {
    CreatePart(pos);

    CurrentPart->Offset = CalcOffset(pos);
    const auto& headersData = hHandler.data();

    CurrentPart->ContentType = headersData.cType();
    CurrentPart->ContentSubtype = headersData.contentSubtype();
    CurrentPart->Name = NDetail::HandleDecodedString(headersData.name());
    CurrentPart->Charset = headersData.charset();
    CurrentPart->Boundary = headersData.boundary();

    CurrentPart->ContentDisposition = headersData.content_disposition();
    CurrentPart->Filename = NDetail::HandleDecodedString(headersData.filename());

    CurrentPart->ContentId = NDetail::HandleContentId(headersData.content_id());
    CurrentPart->Encoding = headersData.content_transfer_encoding();

    if (headersData.from()) {
        CurrentPart->From = std::make_unique<rfc2822::old_rfc2822address>(*headersData.from());
        if (!CurrentPart->From->empty()) {
            const auto& [displayName, address] = CurrentPart->From->front();
            // DARIA-24681: check if we have encoded word instead of address
            if (displayName.empty() &&
                !address.empty() &&
                boost::starts_with(address, boost::as_literal("=?")) &&
                boost::ends_with(address, boost::as_literal("?=")))
            {
                rfc2822::old_rfc2822address addr(NUtil::DecodeHeaderRfc2047(address, CurrentPart->Charset));
                CurrentPart->From->erase(CurrentPart->From->begin());
                CurrentPart->From->splice(CurrentPart->From->begin(), addr, addr.begin(), addr.end());
            }
        }
    }
    if (headersData.to()) {
        CurrentPart->To = std::make_unique<rfc2822::old_rfc2822address>(*headersData.to());
    }
    if (headersData.cc()) {
        CurrentPart->Cc = std::make_unique<rfc2822::old_rfc2822address>(*headersData.cc());
    }

    for (const auto& field: headersData.headerFields()) {
        CurrentPart->Headers.emplace_back(field.name(), boost::trim_left_copy(field.value()));
    }

    return true;
}

template <typename TIterator, typename THHandler>
bool TMessageHandler<TIterator, THHandler>::endPart(const TIterator& pos) {
    if (!CurrentPart) {
        CreatePart(pos);
    }

    CurrentPart->Length = CalcLength(pos); //pos - start - CurrentPart->offset;

    if (boost::iequals(CurrentPart->ContentType, "message") &&
        boost::iequals(CurrentPart->ContentSubtype, "rfc822"))
    {
        ParsePart(MessageData, CurrentPart, InlineLevel);
    }

    if (!PartsStack.empty()) {
        CurrentPart = PartsStack.top();
        PartsStack.pop();
    }
    return true;
}

template <typename TIterator, typename THHandler>
size_t TMessageHandler<TIterator, THHandler>::CalcOffset(const TIterator& pos) const {
    return std::distance(Start, pos) + (Parent ? Parent->Offset : 0);
}

template <typename TIterator, typename THHandler>
size_t TMessageHandler<TIterator, THHandler>::CalcLength(const TIterator& pos) const {
    return std::distance(Start, pos) - (CurrentPart->Offset - (Parent ? Parent->Offset : 0));
}

template <typename TIterator, typename THHandler>
void TMessageHandler<TIterator, THHandler>::CreatePart(const TIterator& pos) {
    if (!CurrentPart) {
        if (Parent) {
            Parent->Children.emplace_back(TPart::CreateShared(MessageData, Parent));
            CurrentPart = Parent->Children.back();
            CurrentPart->Hid = Parent->Hid + "." + std::to_string(Parent->Children.size());
        } else {
            CurrentPart = Message;
            CurrentPart->Hid = "1";
        }
    } else {
        PartsStack.push(CurrentPart);
        CurrentPart->Children.emplace_back(TPart::CreateShared(MessageData, CurrentPart));
        auto newHid = CurrentPart->Hid + "." + std::to_string(CurrentPart->Children.size());
        CurrentPart = CurrentPart->Children.back();
        CurrentPart->Hid = newHid;
    }

    CurrentPart->Offset = CalcOffset(pos);
}

namespace NDetail {

std::string HandleContentId(const std::string& str) {
    auto start = str.begin();
    auto end = str.end();

    while (start != end) {
        if (!isspace(*start)) {
            break;
        }
        ++start;
    }

    if (start != end && *start == '<') {
        ++start;
    }

    auto it = start;
    while (it != end) {
        if (*it == '>' || isspace(*it)) {
            break;
        }
        ++it;
    }

    return std::string(start, it);
}

bool HandlePossiblyIncorrectRfc2231Value(const std::string& value, std::string& result) {
    mulca_mime::DecodedString dstrCopy;
    if (ParseRfc2231Value(value, dstrCopy)) {
        std::string toDecode;
        toDecode.reserve(dstrCopy.contents.size());
        std::string::size_type pos = 0;
        while (pos < dstrCopy.contents.size()) {
            if (dstrCopy.contents[pos] == '%' && pos + 2 < dstrCopy.contents.size()) {
                unsigned char ch;
                char first = dstrCopy.contents[pos + 1];
                char second = dstrCopy.contents[pos + 2];
                if (mulca_mime::get_ext_octet(first, second, ch)) {
                    toDecode += static_cast<char>(ch);
                    pos += 3;
                } else {
                    toDecode += '%';
                    ++pos;
                }
            } else {
                toDecode += dstrCopy.contents[pos];
                ++pos;
            }
        }

        try {
            result = toDecode;
            UTFizer::process(NUtil::GetRecognizer(), dstrCopy.charset, result);
        } catch (...) {
            mulca_mime::convert(dstrCopy.charset.c_str(), "utf-8").doit(toDecode, result);
        }
        return true;
    }

    return false;
}

std::string HandleDecodedString(const mulca_mime::DecodedString& dstr) {
    std::string result;

    if (dstr.contents.empty()) {
        return result;
    }

    if (!dstr.charset.empty()) {
        try {
            result = dstr.contents;
            UTFizer::process(NUtil::GetRecognizer(), dstr.charset, result);
        } catch (...) {
            mulca_mime::convert(dstr.charset.c_str(), "utf-8").doit(dstr.contents, result);
        }
        return result;
    }

    try {
        if (HandlePossiblyIncorrectRfc2231Value(dstr.contents, result)) {
            return result;
        }
    } catch (const std::exception&) {
    }

    const auto encodedWords = mulca_mime::decode_rfc2047(dstr.contents);
    for (const auto& encodedWord: encodedWords) {
        if (encodedWord.word.empty()) {
            continue;
        }

        std::string word = encodedWord.word;
        try {
            UTFizer::process(NUtil::GetRecognizer(), encodedWord.charset, word);
        } catch (...) {
            mulca_mime::convert(encodedWord.charset.c_str(), "utf-8").doit(encodedWord.word, word);
        }
        result += word;
    }

    return mulca_mime::decode_numbered_entities(result);
}

} // namespace NDetail

template class TMessageHandler<const char*>;

} // namespace NNotSoLiteSrv
