#include "rfc2231.h"

#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/phoenix.hpp>

namespace NNotSoLiteSrv {
namespace qi = boost::spirit::qi;

template <typename TIterator>
struct TRfc2231Parser: public boost::spirit::qi::grammar<TIterator, mulca_mime::DecodedString()> {
    TRfc2231Parser(): TRfc2231Parser::base_type(Rfc2231Value) {
        using boost::phoenix::construct;
        using qi::char_;
        using qi::alnum;
        using qi::alpha;
        using qi::space;
        using qi::cntrl;

        Rfc2231Value = ((Charset) >> '\'' >> (Lang) >> '\'' >> (Contents))
            [qi::_val = construct<mulca_mime::DecodedString>(qi::_1, qi::_2, qi::_3, 0)];
        Charset = qi::raw[alnum >> *(*char_("-") >> alnum)];
        Lang %= -(qi::repeat(1, 8)[alpha] >> -('-' >> qi::repeat(1, 8)[alpha]));
        Contents = qi::raw[+(AChar | QChar)];
        AChar = (char_(0x01, 0x7f) - cntrl - char_("*'%") - space - Special);
        QChar = '%' >> qi::repeat(2)[char_("A-Fa-f0-9")];
        Special = char_("()<>@,;:\\\"/][?=");
    }

    qi::rule<TIterator, mulca_mime::DecodedString()> Rfc2231Value;
    qi::rule<TIterator, std::string()> Charset;
    qi::rule<TIterator, std::string()> Lang;
    qi::rule<TIterator, std::string()> Contents;
    qi::rule<TIterator> AChar, QChar, Special;
};

bool ParseRfc2231Value(const std::string& val, mulca_mime::DecodedString& out) {
    auto begin = val.begin();
    auto end = val.end();

    try {
        auto r = qi::parse(
            begin,
            end,
            TRfc2231Parser<std::string::const_iterator>(),
            out);

        return r && begin == end;
    } catch (const std::exception&) {
        return false;
    }
}

} // namespace NNotSoLiteSrv
