#pragma once

#include <mail/notsolitesrv/src/types/common.h>

#include <mimeparser/rfc2822address.h>
#include <util/generic/strbuf.h>

#include <boost/algorithm/string/compare.hpp>

#include <memory>
#include <map>
#include <vector>

namespace NNotSoLiteSrv {

using THeaders = std::vector<std::pair<std::string, std::string>>;
class TPart;
using TPartPtr = std::shared_ptr<TPart>;
using TAddress = std::unique_ptr<rfc2822::old_rfc2822address>;

class TPart {
public:
    using TPartWPtr = std::weak_ptr<TPart>;
    using TParts = std::vector<TPartPtr>;

    TStringBuf GetBody() const { return MessageData.SubStr(Offset, Length); }
    size_t GetOffset() const { return Offset; }
    size_t GetLength() const { return Length; }
    const THid& GetHid() const { return Hid; }
    const std::string& GetContentType() const { return ContentType; }
    const std::string& GetContentSubtype() const { return ContentSubtype; }
    const std::string& GetCharset() const { return Charset; }
    const std::string& GetBoundary() const { return Boundary; }
    const std::string& GetName() const { return Name; }
    const std::string& GetContentDisposition() const { return ContentDisposition; }
    const std::string& GetFilename() const { return Filename; }
    const std::string& GetContentId() const { return ContentId; }
    const std::string& GetEncoding() const { return Encoding; }

    const THeaders& GetHeaders() const { return Headers; }
    const TAddress& GetFrom() const { return From; }
    const TAddress& GetTo() const { return To; }
    const TAddress& GetCc() const { return Cc; }

    const TPartWPtr GetParent() const { return Parent; }
    const TParts& GetChildren() const { return Children; }

    class TConstIterator;
    TConstIterator begin() const;
    TConstIterator end() const;

protected:
    void SetNewParent(TPartPtr parent) {
        for (auto& child: Children) {
            child->Parent = parent;
        }
    }

private:
    template <typename TIter, typename THHandler> friend class TMessageHandler;

    TPart(const TStringBuf& messageData, TPartPtr parent)
        : MessageData(messageData)
        , Parent(parent)
    {}
    static TPartPtr CreateShared(const TStringBuf& messageData, TPartPtr parent);

    size_t Offset = 0;
    size_t Length = 0;
    THid Hid;
    std::string ContentType{"text"};
    std::string ContentSubtype{"plain"};
    std::string Charset{"ascii"};
    std::string Boundary;
    std::string Name;
    std::string ContentDisposition;
    std::string Filename;
    std::string ContentId;
    std::string Encoding;

    THeaders Headers;
    TAddress From;
    TAddress To;
    TAddress Cc;
    TStringBuf MessageData;

    TParts Children;
    TPartWPtr Parent;
};

class TPart::TConstIterator: public std::iterator<std::forward_iterator_tag, const TPart> {
public:
    TConstIterator(): Node(nullptr) {}
    explicit TConstIterator(const TPart* p): Node(p) {}
    bool operator==(const TConstIterator& rhs) const { return Node == rhs.Node; }
    bool operator!=(const TConstIterator& rhs) const { return Node != rhs.Node; }
    const TPart* operator->() const { return Node; }
    const TPart& operator*() const { return *Node; }
    TConstIterator& operator++();

private:
    const TPart* Node = nullptr;
    std::vector<std::pair<const TPart*, TPart::TParts::const_iterator>> NodeStack;
};


} // namespace NNotSoLiteSrv
