#include "processor.h"
#include <mail/notsolitesrv/src/config/message.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/notsolitesrv/src/types/time.h>
#include <mail/notsolitesrv/src/util/headers.h>
#include <mail/notsolitesrv/src/errors.h>
#include <mail/library/utf8/utf8.h>
#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/find.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/asio/ip/host_name.hpp>
#include <boost/spirit/include/qi.hpp>
#include <util/generic/is_in.h>
#include <util/string/cast.h>
#include <set>

namespace NNotSoLiteSrv {
namespace NDetail {

void AddReceivedHeaderToMessage(TContextPtr ctx, TMessagePtr message, const TEnvelope& envelope, const std::string& rcpt) {
    auto [name, value] = NUtil::MakeReceivedHeader(ctx->GetFullSessionId(), envelope, rcpt);
    message->AddHeader(name, value);
}

std::vector<TStringBuf> ExtractXYForwards(const TStringBuf& body) {
    using TIteratorRange = boost::iterator_range<const char*>;
    std::vector<TStringBuf> xyforwards;
    auto begin = body.begin();
    auto end = body.end();
    while (begin != end) {
        TIteratorRange in(begin, end);
        auto res = boost::ifind_first(in, "\nX-Yandex-Forward: ");
        if (!res) {
            break;
        }

        begin = res.end();
        TIteratorRange xyforward;
        bool parsed = boost::spirit::qi::parse(
            res.end(),
            end,
            boost::spirit::qi::raw[+boost::spirit::qi::alnum],
            xyforward);
        if (parsed) {
            xyforwards.emplace_back(xyforward.begin(), xyforward.end());
            end = xyforward.end();
        }
    }

    return xyforwards;
}

} // namespace NDetail

// NB: HERE BE DRAGONS!
// Taken from fastsrv, left for compatibility
void ScanDsnParts(TContextPtr ctx, TMessagePtr message, bool xFailedRecipientsPresent) {
    if (!message->IsAutoSubmitted()) {
        return;
    }

    if (xFailedRecipientsPresent) {
        // A non-standard header X-Failed-Recipients is the only way to safely identify a bounce that
        // comes from mail.ru
        for (const auto& xyf: NDetail::ExtractXYForwards(message->GetBody())) {
            message->AddXYForwardValue(static_cast<std::string>(xyf));
        }
    } else {
        auto iter = std::next(message->begin());
        if (iter == message->end()) {
            return;
        }

        while (iter != message->end()) {
            if (boost::iequals(iter->GetContentType(), "message") &&
                (boost::iequals(iter->GetContentSubtype(), "rfc822") ||
                 boost::iequals(iter->GetContentSubtype(), "delivery-status")))
            {
                for (const auto& xyf: NDetail::ExtractXYForwards(iter->GetBody())) {
                    NSLS_LOG_NOTICE(ctx, logdog::message=
                        std::string("X-Yandex-Forward: ") + static_cast<std::string>(xyf) + " from dsn detected " + "and taken into account");
                    message->AddXYForwardValue(static_cast<std::string>(xyf));
                }
            }
            iter = std::next(iter);
        }
    }
}

TErrorCode Preprocess(
    TContextPtr ctx,
    TMessagePtr message,
    const TEnvelope& envelope,
    NUser::TStoragePtr userStorage,
    bool isHttpSession)
{
    std::string logRcpt;
    if (userStorage->size() == 1) {
        logRcpt = userStorage->begin()->first;
    }
    NDetail::AddReceivedHeaderToMessage(ctx, message, envelope, logRcpt);

    bool retpathPresent = false;
    bool xFailedRecipientsPresent = false;
    std::string dateHeaderValue;
    std::set<std::string> dkimLabels;

    for (const auto& hdr: message->GetHeaders()) {
        const auto name = boost::algorithm::to_lower_copy(hdr.first);

        if (name == "date") {
            dateHeaderValue = hdr.second;
            message->SetDate(hdr.second);
        } else if (name == "return-path") {
            retpathPresent = true;
        } else if (name == "precedence") {
            message->SetPrecedence(hdr.second);
        } else if (name == "auto-submitted") {
            message->SetAutoSubmitted(hdr.second);
        } else if (name == "subject") {
            message->SetSubject(hdr.second);
        } else if (name == "message-id") {
            message->SetMessageId(hdr.second);
        } else if (name == "references") {
            message->SetReferences(hdr.second);
        } else if (name == "in-reply-to") {
            message->SetInReplyTo(hdr.second);
        } else if (name == "authentication-results") {
            message->SetAuthResults(hdr.second);
        } else if (name == "reply-to") {
            message->SetReplyTo(hdr.second);
        } else if (name == "sender") {
            message->SetSender(hdr.second);
        } else if (name == "content-type") {
            if (hdr.second.find("multipart/report") != std::string::npos &&
                hdr.second.find("delivery-status") != std::string::npos)
            {
                message->SetDSN(true);
            }
        } else if (name == "x-auto-reply") {
            message->SetAutoReply(hdr.second);
        } else if (name == "x-failed-recipients") {
            xFailedRecipientsPresent = true;
        } else if (name == "x-yandex-mailish" && !isHttpSession) {
            if (!userStorage->SetMailish(hdr.second)) {
                return EError::UserInvalid;
            }
        } else if (name == "x-yandex-spam") {
            int xyspam = -1;
            TryFromStringWithDefault<int>(TStringBuf{hdr.second}, xyspam, -1);
            message->SetSpamTypeFromXYSpam(xyspam);
        } else if (name == "x-yandex-hint" && !isHttpSession) {
            message->AddXYHint(hdr.second);
        } else if (name == "x-yandex-forward") {
            message->AddXYForwardValue(hdr.second);
        } else if (name == "x-yandex-wdjt-id") {
            dkimLabels.emplace(::NUtil::Utf8Sanitized(hdr.second));
        } else if (name == "x-yandex-uid-status") {
            message->AddSpamTypeByUid(hdr.second);
        } else if (name == "x-yandex-timemark") {
            double startMark;
            if (TryFromString(boost::trim_copy(hdr.second), startMark) && startMark > 0) {
                using namespace NTimeTraits;
                message->SetStartMark(TSystemTimePoint(duration_cast<TSystemDuration>(TFloatSeconds(startMark))));
            }
        } else if (name == "x-yandex-smshost" ||
            name == "x-yandex-folder" ||
            name == "x-yandex-service")
        {
            NSLS_LOG_NOTICE(ctx, logdog::message="skip deprecated header " + hdr.first);
        }

        if (!IsIn(ctx->GetConfig()->Message->RemoveHeaders, name)) {
            message->AddHeader(hdr.first, hdr.second);
        } else if (isHttpSession) {
            NSLS_LOG_WARN(ctx, logdog::message=
                "detect header within HTTP session that should be removed by NwSMTP: " + hdr.first);
        }
    }

    ScanDsnParts(ctx, message, xFailedRecipientsPresent);

    message->CombineXYHints();

    if (dateHeaderValue.empty()) {
        time_t now = time(nullptr);
        auto hint = message->GetXYHintByEmail("");
        if (hint && hint->hdr_date > 0) {
            now = hint->hdr_date;
        }

        message->AddHeader("Date", NUtil::MakeRfc2822Date(now));
    }

    if (!retpathPresent && !envelope.MailFrom.empty()) {
        message->AddHeader("Return-Path", envelope.MailFrom);
    }

    return TErrorCode();
}

} // namespace NNotSoLiteSrv
