#include "meta_save_op.h"

#include <mail/notsolitesrv/src/meta_save_op/util/common.h>
#include <mail/notsolitesrv/src/meta_save_op/util/firstline.h>
#include <mail/notsolitesrv/src/meta_save_op/util/furita.h>
#include <mail/notsolitesrv/src/meta_save_op/util/mdbsave.h>
#include <mail/notsolitesrv/src/meta_save_op/util/mthr.h>
#include <mail/notsolitesrv/src/meta_save_op/util/response.h>
#include <mail/notsolitesrv/src/meta_save_op/util/rules_applier.h>
#include <mail/notsolitesrv/src/meta_save_op/util/tupita.h>
#include <mail/notsolitesrv/src/rules_applier/rules_applier.h>

#include <exception>
#include <string>
#include <utility>
#include <variant>

namespace NNotSoLiteSrv::NMetaSaveOp {

static const std::string META_SAVE_OP{"META_SAVE_OP"};

TMetaSaveOp::TMetaSaveOp(TMetaSaveOpClients clients, TMetaSaveOpComponents components, TContextPtr ctx,
    NUser::TStoragePtr userStorage, boost::asio::io_context& ioContext)
    : Clients(std::move(clients))
    , Components(std::move(components))
    , Ctx(std::move(ctx))
    , UserStorage(std::move(userStorage))
    , IoContext(ioContext)
{
}

void TMetaSaveOp::SetOpParams(TRequest request, TMetaSaveOpCallback callback) {
    Params.Request = std::move(request);
    Params.Callback = std::move(callback);
}

#include <yplatform/yield.h>

void TMetaSaveOp::operator()(TYieldCtx yieldCtx, TErrorCode errorCode, TMetaSaveOp::TResult result) {
    if (Ctx->GetConfig()->MetaSaveOp->DropAsync && (!Params.Request.sync_dlv)) {
        SetUsersError(EError::MetaSaveOpError);
        return Params.Callback(EError::MetaSaveOpError);
    }

    reenter(yieldCtx) {
        yield RequestFirstline(yieldCtx);
        if (!ProcessFirstlineResult(std::move(errorCode), std::move(result))) {
            yield break;
        }

        if (FiltersInUse(Params.Request.recipients)) {
            FuritaUids = MakeFuritaUids(Params.Request.recipients);
            while (!FuritaUids.empty()) {
                Uid = FuritaUids.extract(FuritaUids.cbegin()).value();
                yield RequestFurita(yieldCtx);
                ProcessFuritaListResult(std::move(errorCode), std::move(result));
                if (ClientResults.Furita[Uid].Result &&
                    EnabledRuleWithVerifiedActionAvailable(ClientResults.Furita[Uid].Result->Rules))
                {
                    yield RequestTupita(yieldCtx);
                    ProcessTupitaCheckResult(std::move(errorCode), std::move(result));
                }
            }

            ComponentResults.RulesApplier.Result = NRulesApplier::ApplyRules(MakeRulesApplierRequest(
                Params.Request), ClientResults.Furita, ClientResults.Tupita);
        }

        if (RecipientToSaveForAvailable(Params.Request.recipients, ComponentResults.RulesApplier.Result)) {
            if (!ProcessMthrResult(RequestMthr())) {
                yield break;
            }

            yield RequestMdbSave(yieldCtx);
            if (!ProcessMdbSaveResult(std::move(errorCode), std::move(result))) {
                yield break;
            }
        }
    }

    if (yieldCtx.is_complete()) {
        ProcessResponses();
    }
}

#include <yplatform/unyield.h>

void TMetaSaveOp::RequestFirstline(const TYieldCtx& yieldCtx) {
    Components.Firstline->Firstline(Ctx, MakeFirstlineRequest(Params.Request), yieldCtx, IoContext);
}

void TMetaSaveOp::RequestFurita(const TYieldCtx& yieldCtx) {
    Clients.Furita->List(IoContext, Uid, yieldCtx);
}

void TMetaSaveOp::RequestTupita(const TYieldCtx& yieldCtx) {
    auto checkRequest{MakeTupitaCheckRequest(Params.Request, ComponentResults.Firstline.Result, Uid,
        ClientResults.Furita[Uid].Result->Rules)};
    Clients.Tupita->Check(IoContext, Uid, Params.Request.session_id, std::move(checkRequest), yieldCtx);
}

void TMetaSaveOp::RequestMdbSave(const TYieldCtx& yieldCtx) {
    auto mdbSaveRequest = MakeMdbSaveRequest(Params.Request, ComponentResults.Firstline.Result,
        *ComponentResults.Mthr.Result, ComponentResults.RulesApplier.Result);
    Clients.MdbSave->MdbSave(IoContext, Params.Request.session_id, std::move(mdbSaveRequest), yieldCtx);
}

bool TMetaSaveOp::ProcessFirstlineResult(TErrorCode errorCode, TResult result) {
    if (errorCode) {
        ComponentResults.Firstline.ErrorCode = errorCode;
        NSLS_LOG_CTX_ERROR(log::message = "failed to get firstline",
            log::error_code = ComponentResults.Firstline.ErrorCode, log::where_name = META_SAVE_OP);
        return false;
    }

    if (!CheckSpecificResult<NFirstline::TFirstlineResult>(result)) {
        ComponentResults.Firstline.ErrorCode = EError::MetaSaveOpIncorrectResult;
        NSLS_LOG_CTX_ERROR(log::message = "failed to get firstline result",
            log::error_code = ComponentResults.Firstline.ErrorCode, log::where_name = META_SAVE_OP);
        return false;
    }

    ComponentResults.Firstline.Result = std::get<NFirstline::TFirstlineResult>(std::move(result));
    return true;
}

void TMetaSaveOp::ProcessFuritaListResult(TErrorCode errorCode, TResult result) {
    if (errorCode) {
        ClientResults.Furita[Uid].ErrorCode = std::move(errorCode);
        NSLS_LOG_CTX_ERROR(log::message = "failed to call furita", log::uid = std::to_string(Uid),
            log::error_code = ClientResults.Furita[Uid].ErrorCode, log::where_name = META_SAVE_OP);
    } else if (!CheckSpecificResult<NFurita::TListResult>(result)) {
        ClientResults.Furita[Uid].ErrorCode = EError::MetaSaveOpIncorrectResult;
        NSLS_LOG_CTX_ERROR(log::message = "failed to get furita result", log::uid = std::to_string(Uid),
            log::error_code = ClientResults.Furita[Uid].ErrorCode, log::where_name = META_SAVE_OP);
    } else {
        ClientResults.Furita[Uid].Result = std::get<NFurita::TListResult>(std::move(result));
    }
}

void TMetaSaveOp::ProcessTupitaCheckResult(TErrorCode errorCode, TResult result) {
    if (errorCode) {
        ClientResults.Tupita[Uid].ErrorCode = std::move(errorCode);
        NSLS_LOG_CTX_ERROR(log::message = "failed to call tupita", log::uid = std::to_string(Uid),
            log::error_code = ClientResults.Tupita[Uid].ErrorCode, log::where_name = META_SAVE_OP);
    } else if (!CheckSpecificResult<NTupita::TCheckResult>(result)) {
        ClientResults.Tupita[Uid].ErrorCode = EError::MetaSaveOpIncorrectResult;
        NSLS_LOG_CTX_ERROR(log::message = "failed to get tupita result", log::uid = std::to_string(Uid),
            log::error_code = ClientResults.Tupita[Uid].ErrorCode, log::where_name = META_SAVE_OP);
    } else {
        ClientResults.Tupita[Uid].Result = std::get<NTupita::TCheckResult>(std::move(result));
        if (ClientResults.Tupita[Uid].Result->Result.size() != 1) {
            NSLS_LOG_CTX_ERROR(log::message = "invalid tupita result", log::uid = std::to_string(Uid),
                log::where_name = META_SAVE_OP);
        }
    }
}

bool TMetaSaveOp::ProcessMdbSaveResult(TErrorCode errorCode, TResult result) {
    if (errorCode) {
        ClientResults.MdbSave.ErrorCode = errorCode;
        NSLS_LOG_CTX_ERROR(log::message = "failed to call mdbsave",
            log::error_code = ClientResults.MdbSave.ErrorCode, log::where_name = META_SAVE_OP);
        return false;
    }

    if (!CheckSpecificResult<NMdbSave::TMdbSaveResult>(result)) {
        ClientResults.MdbSave.ErrorCode = EError::MetaSaveOpIncorrectResult;
        NSLS_LOG_CTX_ERROR(log::message = "failed to get mdbsave result",
            log::error_code = ClientResults.MdbSave.ErrorCode, log::where_name = META_SAVE_OP);
        return false;
    }

    ClientResults.MdbSave.Result = std::get<NMdbSave::TMdbSaveResult>(std::move(result));
    return true;
}

NMthr::TMthrResult TMetaSaveOp::RequestMthr() const {
    return Components.Mthr->GetThreadInfo(Ctx, MakeMthrRequest(Params.Request));
}

bool TMetaSaveOp::ProcessMthrResult(NMthr::TMthrResult result) {
    if (!result) {
        ComponentResults.Mthr.ErrorCode = EError::MthrError;
        NSLS_LOG_CTX_ERROR(log::message = "failed to get thread info",
            log::error_code = ComponentResults.Mthr.ErrorCode, log::where_name = META_SAVE_OP);
        return false;
    }

    ComponentResults.Mthr.Result = std::move(result);
    return true;
}

void TMetaSaveOp::ProcessResponses() const {
    TErrorCode errorCode;
    try {
        if (FatalErrorPresent()) {
            NSLS_LOG_CTX_ERROR(log::message = "status=error", log::where_name = META_SAVE_OP);
            errorCode = EError::MetaSaveOpError;
            SetUsersError(errorCode);
        } else {
            NSLS_LOG_CTX_NOTICE(log::message = "status=ok", log::where_name = META_SAVE_OP);
            const auto response{MakeResponse(Params.Request.recipients, ComponentResults.RulesApplier.Result,
                ClientResults.MdbSave.Result)};
            UpdateRecipientsStatusFromResponse(Ctx, *UserStorage, response);
        }
    } catch (const std::exception& ex) {
        using namespace std::string_literals;
        NSLS_LOG_CTX_ERROR(log::message = "failed to process responses ("s + ex.what() + ")",
            log::where_name = META_SAVE_OP);
        errorCode = EError::MetaSaveOpError;
    }

    Params.Callback(std::move(errorCode));
}

bool TMetaSaveOp::FatalErrorPresent() const {
    return FatalClientErrorPresent() || FatalComponentErrorPresent();
}

bool TMetaSaveOp::FatalClientErrorPresent() const {
    return static_cast<bool>(ClientResults.MdbSave.ErrorCode);
}

bool TMetaSaveOp::FatalComponentErrorPresent() const {
    return
        ComponentResults.Firstline.ErrorCode ||
        ComponentResults.Mthr.ErrorCode;
}

void TMetaSaveOp::SetUsersError(const TErrorCode& errorCode) const {
    auto filter = [](const auto& user) {
        return
            (user.Status == NUser::ELoadStatus::Found) &&
            user.DeliveryParams.NeedDelivery &&
            (!user.DeliveryResult.ErrorCode);
    };

    auto users = UserStorage->GetFilteredUsers(std::move(filter));
    for (auto& element: users) {
        element.second.DeliveryResult.ErrorCode = errorCode;
    }
}

}
