#include "request.h"
#include <mail/notsolitesrv/src/config/message.h>
#include <mail/notsolitesrv/src/message/part.h>
#include <mail/notsolitesrv/src/message/types.h>
#include <mail/notsolitesrv/src/tskv/logger.h>
#include <mail/notsolitesrv/src/util/headers.h>
#include <mail/notsolitesrv/src/util/recognizer.h>
#include <mail/notsolitesrv/src/util/string.h>
#include <mail/notsolitesrv/src/errors.h>
#include <mail/library/utf8/utf8.h>
#include <mail/butil/include/butil/butil.h>
#include <mail/mail_getter/include/mail_getter/UTFizer.h>
#include <mail/message_types/lib/message_types.h>
#include <mail/mimeparser/include/mimeparser/ccnv.h>
#include <mail/mimeparser/include/mimeparser/encoding.h>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/split.hpp>
#include <util/generic/algorithm.h>
#include <util/generic/is_in.h>
#include <util/generic/ylimits.h>
#include <util/string/cast.h>
#include <algorithm>

namespace NNotSoLiteSrv::NMetaSaveOp {

const std::string HAMON_LABEL = "hamon_label";

bool THidComparator::operator()(const THid& lhs, const THid& rhs) const {
    std::vector<std::string> lhsParts, rhsParts;
    boost::split(lhsParts, lhs, boost::is_any_of("."), boost::token_compress_on);
    boost::split(rhsParts, rhs, boost::is_any_of("."), boost::token_compress_on);

    return std::lexicographical_compare(
        lhsParts.begin(), lhsParts.end(),
        rhsParts.begin(), rhsParts.end(),
        [](const auto& l, const auto& r) {
            return std::stoul(l) < std::stoul(r);
        });
}

inline std::string TrimSanitized(const std::string& val) {
    return NUtil::StripBadChars(::NUtil::Utf8Trim(::NUtil::Utf8Sanitized(val)));
}

inline std::string Sanitized(const std::string& val) {
    return NUtil::StripBadChars(::NUtil::Utf8Sanitized(val));
}

boost::optional<TEmailAddress> ToEmailAddress(TContextPtr ctx, const rfc2822::address_pair_t& addr) {
    TEmailAddress ret;

    if (!NUtil::SplitEmail(addr.second, ret.Local, ret.Domain)) {
        ret.Local = TrimSanitized(addr.second);

        std::string msg = "split email returned false. ";
        msg += "display_name: \"" + addr.first.substr(0, 4096) + "\"; ";
        msg += "email: \"" + addr.second.substr(0, 4096) + "\"; ";
        NSLS_LOG_NOTICE(ctx,
            logdog::message=std::move(msg),
            logdog::where_name="to_email_address_not_splitted");
    } else {
        ret.Local = TrimSanitized(ret.Local);
        ret.Domain = TrimSanitized(ret.Domain);
    }

    ret.DisplayName =
        NUtil::StripBadChars(
        ::NUtil::Utf8Backslash(
        ::NUtil::Utf8Unquote(
        ::NUtil::Utf8Trim(
        ::NUtil::Utf8Sanitized(
        NUtil::DeBackslash(
        NUtil::DecodeHeaderRfc2047(addr.first)))))));
    return ret;
}

boost::optional<TEmailAddress> ToEmailAddress(const rfc2822::address_pair_t& addr) {
    TEmailAddress ret;

    if (!NUtil::SplitEmail(addr.second, ret.Local, ret.Domain)) {
        ret.Local = TrimSanitized(addr.second);
    } else {
        ret.Local = TrimSanitized(ret.Local);
        ret.Domain = TrimSanitized(ret.Domain);
    }

    ret.DisplayName =
        NUtil::StripBadChars(
        ::NUtil::Utf8Backslash(
        ::NUtil::Utf8Unquote(
        ::NUtil::Utf8Trim(
        ::NUtil::Utf8Sanitized(
        NUtil::DeBackslash(
        NUtil::DecodeHeaderRfc2047(addr.first)))))));
    return ret;
}

boost::optional<TEmailAddress> ToEmailAddress(const std::string& straddr) {
    try {
        rfc2822::old_rfc2822address addressList(straddr);
        if (addressList.ok() && !addressList.empty()) {
            for (const auto& addr: addressList) {
                if (!addr.first.empty() || !addr.second.empty()) {
                    auto parsed = ToEmailAddress(addr);
                    if (parsed &&
                        (!parsed->Local.empty() ||
                            !parsed->Domain.empty() ||
                            !parsed->DisplayName.empty()))
                    {
                        return parsed;
                    }
                }
            }
        }
    } catch (const std::exception&) {
        // ignore
    }
    return {};
}

template <typename TContainer, typename TOutputIterator>
void FillAddressList(const TContainer& from, TOutputIterator to) {
    for (const auto& addr: from) {
        auto email = ToEmailAddress(addr);
        if (!email) {
            continue;
        }
        *to++ = *email;
    }
}

template <typename TContainer, typename TOutputIterator>
void FillAddressList(TContextPtr ctx, const TContainer& from, TOutputIterator to) {
    for (const auto& addr: from) {
        auto email = ToEmailAddress(ctx, addr);
        if (!email) {
            continue;
        }
        *to++ = *email;
    }
}

size_t GetPartOffset(off_t originalOffset, off_t offsetDiff) {
    Y_ENSURE(originalOffset > offsetDiff);
    Y_ENSURE(originalOffset - offsetDiff > 0);
    Y_ENSURE(Max<off_t>() - originalOffset > -offsetDiff);

    return originalOffset - offsetDiff;
}

TPart CreatePartFromMessagePart(const NNotSoLiteSrv::TPart& msgPart, off_t offsetDiff) {
    TPart part;

    part.content_type = TrimSanitized(msgPart.GetContentType());
    part.content_subtype = TrimSanitized(msgPart.GetContentSubtype());
    part.charset = TrimSanitized(msgPart.GetCharset());
    part.encoding = TrimSanitized(msgPart.GetEncoding());
    part.offset = GetPartOffset(msgPart.GetOffset(), offsetDiff);
    part.length = msgPart.GetLength();
    if (!msgPart.GetBoundary().empty()) {
        part.boundary = TrimSanitized(msgPart.GetBoundary());
    }
    if (!msgPart.GetName().empty()) {
        part.name = Sanitized(msgPart.GetName());
    }
    if (!msgPart.GetContentDisposition().empty()) {
        part.content_disposition = TrimSanitized(msgPart.GetContentDisposition());
    }
    if (!msgPart.GetFilename().empty()) {
        part.file_name = Sanitized(msgPart.GetFilename());
    }
    if (!msgPart.GetContentId().empty()) {
        part.content_id = Sanitized(msgPart.GetContentId());
    }

    return part;
}

void FindAndUpdateBestTextPart(const TMessagePtr message, TPartMap& parts) {
    struct TBestTextPart {
        ssize_t HidDotCount = Max(); // less dots is better
        std::string Hid;
        bool IsHtml = false;
    };
    TBestTextPart bestTextPart;

    for (const auto& msgPart: *message) {
        if (!message->IsPartApplicableForDB(msgPart)) {
            continue;
        }

        std::string fname = msgPart.GetFilename().empty() ? msgPart.GetName() : msgPart.GetFilename();

        bool isHtmlPart = boost::iequals(msgPart.GetContentSubtype(), "html");
        if (!boost::iequals(fname, "smime.p7s") &&
            boost::iequals(msgPart.GetContentType(), "text") &&
            (isHtmlPart || boost::iequals(msgPart.GetContentSubtype(), "plain")))
        {
            auto hidDotCount = Count(msgPart.GetHid(), '.');
            if (hidDotCount < bestTextPart.HidDotCount ||
                (hidDotCount == bestTextPart.HidDotCount &&
                    isHtmlPart &&
                    !bestTextPart.IsHtml))
            {
                bestTextPart.HidDotCount = hidDotCount;
                bestTextPart.Hid = msgPart.GetHid();
                bestTextPart.IsHtml = isHtmlPart;
            }
        }
    }

    if (!bestTextPart.Hid.empty() && IsIn(parts, bestTextPart.Hid)) {
        auto& part = parts[bestTextPart.Hid];
        auto msgPartIter = FindIf(
            *message,
            [&bestTextPart](const auto& msgPart) {
                return msgPart.GetHid() == bestTextPart.Hid;
            });

        std::string data;
        if (boost::iequals(msgPartIter->GetEncoding(), "base64")) {
            auto body = msgPartIter->GetBody();
            data = decode_base64({body.begin(), body.end()});
        } else if (boost::iequals(msgPartIter->GetEncoding(), "quoted-printable")) {
            auto body = msgPartIter->GetBody();
            data = decode_qp({body.begin(), body.end()});
        } else {
            data = msgPartIter->GetBody();
        }
        UTFizer::process(NUtil::GetRecognizer(), msgPartIter->GetCharset(), data);
        part.data = ::NUtil::Utf8Sanitized(data);
    }
}

void FindPartsAndAttachments(
    const TMessagePtr message,
    TPartMap& parts,
    TAttachmentMap& attachments)
{
    using namespace NUtil;
    for (const auto& part: *message) {
        if (message->IsPartApplicableForDB(part)) {
            parts.emplace(part.GetHid(), CreatePartFromMessagePart(part, message->GetOffsetDiff()));
        }
    }

    FindAndUpdateBestTextPart(message, parts);

    for (const auto& attach: message->GetAttachments()) {
        attachments.emplace(
            attach.first,
            TAttachment{attach.second.Name, attach.second.Type, attach.second.Size});
    }
}

TEnvelope CreateEnvelope(
    TMessagePtr message,
    const NNotSoLiteSrv::TEnvelope& envelope,
    const NUser::TStorage& userStorage)
{
    TEnvelope env;
    env.remote_ip = envelope.RemoteIp;
    env.remote_host = envelope.RemoteHost;
    env.helo = envelope.Lhlo;
    env.mail_from = envelope.MailFrom;
    env.received_date = message->GetReceivedDate();
    for (const auto& [rcptto, user]: userStorage.GetFilteredUsers(NUser::FromRcptTo)) {
        env.recipients.push_back(rcptto);
    }

    return env;
}

TMessage CreateMessage(TContextPtr ctx, TMessagePtr origMessage) {
    TMessage msg;

    msg.subject = Sanitized(origMessage->GetSubject());
    msg.message_id = origMessage->GetMessageId();
    msg.size = origMessage->GetFinalLength();
    msg.spam = origMessage->IsSpam();
    if (origMessage->GetFrom()) {
        FillAddressList(*origMessage->GetFrom(), std::back_inserter(msg.from));
    }
    if (origMessage->GetTo()) {
        FillAddressList(ctx, *origMessage->GetTo(), std::back_inserter(msg.to));
    }
    if (origMessage->GetCc()) {
        msg.cc.emplace();
        FillAddressList(*origMessage->GetCc(), std::back_inserter(*msg.cc));
    }
    if (!origMessage->GetSender().empty()) {
        msg.sender = origMessage->GetSender();
    }
    msg.reply_to = origMessage->GetReplyTo();
    if (!origMessage->GetInReplyTo().empty()) {
        msg.in_reply_to = origMessage->GetInReplyTo();
    }

    const auto& refs = origMessage->GetReferences();
    if (!refs.empty()) {
        msg.references = refs;
    }

    TAttachmentMap attachments;
    FindPartsAndAttachments(origMessage, msg.parts, attachments);
    if (!attachments.empty()) {
        msg.attachments = attachments;
    }

    msg.date = origMessage->GetDate();

    return msg;
}

std::pair<TFolder, ENoSuchFolderAction> CreateFolder(const TXYandexHint& hint, bool isSpam, bool isMailish) {
    TFolder folder;
    ENoSuchFolderAction noSuchFolderAction = ENoSuchFolderAction::FallbackToInbox;
    if (hint.save_to_sent != boost::none) {
        folder.path.emplace();
        folder.path->path = "\\Sent";
        noSuchFolderAction = ENoSuchFolderAction::Fail;
    } else if (!hint.fid.empty() && boost::all(hint.fid, &isdigit)) {
        folder.fid = hint.fid;
        // TODO(MAILDLV-4960) replace all conditions with sync_dlv
        noSuchFolderAction = (hint.imap || isMailish || hint.no_such_fid_fail) ?
            ENoSuchFolderAction::Fail : ENoSuchFolderAction::FallbackToInbox;
    } else {
        folder.path.emplace();

        if (!hint.folder_path.empty() && !isSpam) {
            folder.path->path = hint.folder_path;
            if (!hint.folder_path_delim.empty()) {
                folder.path->delim = hint.folder_path_delim;
            }
            noSuchFolderAction = ENoSuchFolderAction::Create;
        } else if (isSpam && !hint.folder_spam_path.empty()) {
            folder.path->path = hint.folder_spam_path;
            if (!hint.folder_path_delim.empty()) {
                folder.path->delim = hint.folder_path_delim;
            }
            noSuchFolderAction = ENoSuchFolderAction::Create;
        } else if (!hint.folder.empty()) {
            folder.path->path = hint.folder;
            folder.path->delim = "|";
            noSuchFolderAction = ENoSuchFolderAction::FallbackToInbox;
        } else if (isSpam) {
            folder.path->path = "\\Spam";
            noSuchFolderAction = ENoSuchFolderAction::Fail;
        } else {
            folder.path->path = "\\Inbox";
            noSuchFolderAction = ENoSuchFolderAction::Fail;
        }
    }

    return {folder, noSuchFolderAction};
}

enum EMixedFlags {
    MIXED_ATTACHED   = 0x00001,
    MIXED_SPAM       = 0x00004,
    MIXED_POSTMASTER = 0x00010,
    MIXED_RECENT     = 0x00020,
    MIXED_DRAFT      = 0x00040,
    MIXED_DELETED    = 0x00080,
    MIXED_FORWARDED  = 0x00200,
    MIXED_ANSWERED   = 0x00400,
    MIXED_SEEN       = 0x00800,
    MIXED_APPEND     = 0x10000
};

void UpdateLabelSymbolsFromMixed(std::vector<std::string>& labelSymbols, int64_t mixed) {
    if (mixed & MIXED_ATTACHED) {
        labelSymbols.emplace_back("attached_label");
    }
    if (mixed & MIXED_SPAM) {
        labelSymbols.emplace_back("spam_label");
    }
    if (mixed & MIXED_POSTMASTER) {
        labelSymbols.emplace_back("postmaster_label");
    }
    if (mixed & MIXED_RECENT) {
        labelSymbols.emplace_back("recent_label");
    }
    if (mixed & MIXED_DRAFT) {
        labelSymbols.emplace_back("draft_label");
    }
    if (mixed & MIXED_DELETED) {
        labelSymbols.emplace_back("deleted_label");
    }
    if (mixed & MIXED_FORWARDED) {
        labelSymbols.emplace_back("forwarded_label");
    }
    if (mixed & MIXED_ANSWERED) {
        labelSymbols.emplace_back("answered_label");
    }
    if (mixed & MIXED_SEEN) {
        labelSymbols.emplace_back("seen_label");
    }
    if (mixed & MIXED_APPEND) {
        labelSymbols.emplace_back("append_label");
    }
}

void AddSoSystemType(
    TContextPtr ctx,
    const std::string& typeName,
    bool isSpam,
    TParams& params,
    std::string& messageTypes)
{
    if (boost::iequals(typeName, "hamon")) {
        // Special hamon hack
        // There is no way to get labels from SO, but only types.
        // So we use hack here to convert type to label.
        // see: https://st.yandex-team.ru/SODEV-2069#5ef0c94040e3f14a5e7cfa14
        if (isSpam) {
            NSLS_LOG_NOTICE(ctx, logdog::message="skip adding 'hamon' type because of spam");
        } else {
            params.label_symbols.emplace_back(HAMON_LABEL);
            NSLS_LOG_NOTICE(ctx, logdog::message="converted 'hamon' type to 'symbol:" + HAMON_LABEL + "'");
        }
        return;
    }

    auto resolvedName = ConvertSoLabelToType(typeName);
    if (!resolvedName.empty()) {
        params.labels.emplace_back(resolvedName, "so");
        if (!messageTypes.empty()) {
            messageTypes.append(", ");
        }
        messageTypes.append(typeName);

        if (!params.tab && boost::istarts_with(typeName, "t_")) {
            params.tab = typeName.substr(sizeof("t_") - 1);
        }
    }
}

void UpdateLabelsInParams(
    TContextPtr ctx,
    TParams& params,
    const TXYandexHint& hint,
    bool isSpam,
    bool hasAttachments,
    bool sharedStid)
{
    std::string messageTypes;
    for (const auto& label: hint.label) {
        if (boost::istarts_with(label, "SystMetkaSO:")) {
            auto typeName = label.substr(sizeof("SystMetkaSO:") - 1);
            AddSoSystemType(ctx, typeName, isSpam, params, messageTypes);
        } else if (boost::istarts_with(label, "domain_")) {
            auto name = label.substr(sizeof("domain_") - 1);
            NSLS_LOG_DEBUG(ctx, logdog::message="added label " + name + " (so)");
            params.labels.emplace_back(name, "social");
        } else if (boost::istarts_with(label, "symbol:")) {
            params.label_symbols.emplace_back(label.substr(sizeof("symbol:") - 1));
        } else {
            params.labels.emplace_back(label, "system");
        }
    }
    NSLS_LOG_NOTICE(ctx, logdog::message="message types: " + messageTypes);

    for (const auto& label: hint.userlabel) {
        params.labels.emplace_back(label, "user");
    }

    for (const auto& label: hint.imaplabel) {
        params.labels.emplace_back(label, "imap");
    }

    // Fill special label symbols
    if (isSpam) {
        params.label_symbols.emplace_back("spam_label");
    }
    if (hasAttachments) {
        params.label_symbols.emplace_back("attached_label");
    }
    if (sharedStid || hint.copy_to_inbox) {
        params.label_symbols.emplace_back("mulcaShared_label");
    }
    if (!isSpam && hint.priority_high) {
        params.label_symbols.emplace_back("important_label");
    }
    if (hint.imap) {
        params.label_symbols.emplace_back("append_label");
    }
    if (hint.mixed) {
        UpdateLabelSymbolsFromMixed(params.label_symbols, hint.mixed);
    }
    if (IsIn(params.label_symbols, "mute_label") && !IsIn(params.label_symbols, "seen_label")) {
        params.label_symbols.emplace_back("seen_label");
    }

    for (const auto& lid: hint.lid) {
        uint64_t tmp;
        if (TryFromString(lid, tmp)) {
            params.lids.emplace_back(lid);
        }
    }

    SortUnique(params.lids);
    SortUniqueBy(params.labels, [](const auto& v) { return std::tie(v.name, v.type); });
    SortUnique(params.label_symbols);
}

bool IsSubscription(const TXYandexHint& hint, const std::set<NMail::EMessageType>& subscriptions) {
    return !subscriptions.empty() && std::any_of(hint.label.cbegin(), hint.label.cend(), [&](const auto& label) {
        if (boost::istarts_with(label, "SystMetkaSO:")) {
            NMail::EMessageType type = NMail::MT_UNDEFINED;
            return (TryFromString(label.substr(sizeof("SystMetkaSO:") - 1), type) && subscriptions.contains(type));
        }
        return false;
    });
}

TUser CreateUser(const std::string& email, const NUser::TUser& recipient) {
    TUser user;
    user.org_id = recipient.OrgId;
    user.uid = std::stoull(recipient.Uid);
    if (!recipient.Suid.empty()) {
        user.suid = recipient.Suid;
    }
    if (auto parsedEmail = ToEmailAddress(email)) {
        user.email = *parsedEmail;
    }
    user.country = recipient.Country;
    user.is_phone_confirmed = recipient.IsPhoneConfirmed;
    user.born_date = recipient.RegistrationDate;
    user.karma = recipient.Karma;
    user.karma_status = recipient.KarmaStatus;

    return user;
}

TParams CreateParams(
    TContextPtr ctx,
    const NUser::TUser& user,
    const TMessagePtr& message,
    bool isMailish,
    bool hasAttachments,
    bool sharedStid)
{
    auto hint = message->GetXYHintByUid(user.Uid);
    bool isSpam = message->IsSpam(user.Uid);
    auto personalStid = message->GetPersonalStid(user.Uid);
    TParams params;

    NSLS_LOG_NOTICE(ctx, logdog::message=ToString(message->GetSpamResolutionType(user.Uid)) +
        " spam resolution used for UID: " + user.Uid + ", SUID: " + user.Suid);

    params.use_domain_rules = user.DeliveryParams.UseDomainRules;
    params.use_filters = hint.filters;
    params.mailish = isMailish;
    params.ignore_duplicates = params.mailish || hint.imap;
    params.disable_push = hint.disable_push;
    params.store_as_deleted = (hint.save_to_sent == false || hint.store_as_deleted);
    params.spam = isSpam;
    params.imap = hint.imap;
    if (hint.external_imap_id != 0) {
        params.external_imap_id = hint.external_imap_id;
    }
    if (!hint.bcc.empty()) {
        params.bcc.emplace();
        FillAddressList(hint.bcc, std::back_inserter(*params.bcc));
    }
    if (!hint.mid.empty()) {
        params.old_mid = hint.mid;
    }
    if (personalStid) {
        params.stid = personalStid;
        params.offset_diff = message->GetPersonalHeadersLength(user.Uid);
    }

    if (params.use_filters) {
        params.process_as_subscription = IsSubscription(hint, ctx->GetConfig()->MSearch->MessageTypes);
    }

    std::tie(params.folder, params.no_such_folder_action) = CreateFolder(hint, isSpam, isMailish);
    UpdateLabelsInParams(ctx, params, hint, isSpam, hasAttachments, personalStid ? false : sharedStid);

    return params;
}

TRecipient CloneAsCopyToInboxRcpt(TContextPtr ctx, const TRecipient& rcpt) {
    TRecipient ctiRcpt;
    ctiRcpt.user = rcpt.user;
    ctiRcpt.params.bcc = rcpt.params.bcc;
    ctiRcpt.params.labels = rcpt.params.labels;
    ctiRcpt.params.lids = rcpt.params.lids;
    std::copy_if(
        rcpt.params.label_symbols.begin(),
        rcpt.params.label_symbols.end(),
        std::back_inserter(ctiRcpt.params.label_symbols),
        [&ctx](const auto& symbol) {
            return !IsIn(ctx->GetConfig()->Message->SkipSymbolsForCopyToInbox, symbol);
        });

    ctiRcpt.params.tab = rcpt.params.tab;
    ctiRcpt.params.stid = rcpt.params.stid;
    ctiRcpt.params.offset_diff = rcpt.params.offset_diff;
    ctiRcpt.params.mailish = rcpt.params.mailish;
    ctiRcpt.params.disable_push = rcpt.params.disable_push;
    ctiRcpt.params.spam = rcpt.params.spam;
    return ctiRcpt;
}

TRecipientMap CreateRecipients(TContextPtr ctx, NUser::TStorage& userStorage, TMessagePtr message,
    bool hasAttachments, bool updateOuterData)
{
    TRecipientMap recipients;
    int rid = 1;
    auto users = userStorage.GetFilteredUsers(
        [](const NUser::TUser& user) {
            return user.DeliveryParams.NeedDelivery &&
                user.Status == NUser::ELoadStatus::Found &&
                user.DeliveryParams.DeliveryId.empty() &&
                !user.DeliveryResult.ErrorCode;
        });

    bool sharedStid = (users | [message](const NUser::TUser& u){ return !message->HasPersonalHeaders(u.Uid); }).size() > 1;
    std::set<std::string> seenUids;

    for (auto& [email, user]: users) {
        // store message only once per user (copy_to_inbox handled below)
        if (IsIn(seenUids, user.Uid)) {
            continue;
        }

        seenUids.emplace(user.Uid);

        const auto rcptUser = CreateUser(email, user);
        const auto params = CreateParams(ctx, user, message, userStorage.IsMailish(), hasAttachments,
            sharedStid);

        const TRecipient tmpRcpt{rcptUser, params};
        auto rcpt = tmpRcpt;

        const auto hint = message->GetXYHintByUid(user.Uid);
        if (hint.save_to_sent && *hint.save_to_sent) {
            // MAILDLV-4921 Remove hamon_label from Sent, keep it for Inbox
            auto hamonLabelIter = Find(rcpt.params.label_symbols, HAMON_LABEL);
            if (hamonLabelIter != rcpt.params.label_symbols.end()) {
                rcpt.params.label_symbols.erase(hamonLabelIter);
                NSLS_LOG_NOTICE(ctx,
                    logdog::message="symbol:" + HAMON_LABEL + " removed for " +
                        std::to_string(rcpt.user.uid) + " for Sent");
            }
        }
        const auto rcptId = std::to_string(rid++);
        recipients.emplace(rcptId, rcpt);
        if (updateOuterData) {
            user.DeliveryParams.DeliveryId = rcptId;
        }

        if (hint.copy_to_inbox) {
            auto ctiRcpt = CloneAsCopyToInboxRcpt(ctx, tmpRcpt);
            if (ctiRcpt.params.use_filters) {
                ctiRcpt.params.process_as_subscription = IsSubscription(hint,
                    ctx->GetConfig()->MSearch->MessageTypes);
            }
            auto ctiRcptId = std::to_string(rid++);
            recipients.emplace(ctiRcptId, ctiRcpt);
            if (updateOuterData) {
                userStorage.CloneAsCopyToInboxUser(email, user, ctiRcptId);
            }
        }
    }

    return recipients;
}

bool HasAttachments(const TMessage& message) {
    return message.attachments && !message.attachments->empty();
}

TRequest::TRequest(
    TContextPtr ctx,
    TMessagePtr origMessage,
    const NNotSoLiteSrv::TEnvelope& smtpEnvelope,
    NUser::TStorage& userStorage,
    bool updateOuterData)
{
    session_id = ctx->GetFullSessionId();
    stid = origMessage->GetStid();

    envelope = CreateEnvelope(origMessage, smtpEnvelope, userStorage);
    message = CreateMessage(ctx, origMessage);
    recipients = CreateRecipients(ctx, userStorage, origMessage, HasAttachments(message), updateOuterData);

    if (auto defaultHint = origMessage->GetXYHint()) {
        sync_dlv = defaultHint->sync_dlv;
        for (const auto& label: defaultHint->label) {
            if (!boost::istarts_with(label, "SystMetkaSO:")) {
                continue;
            }

            auto name = label.substr(sizeof("SystMetkaSO:") - 1);
            NMail::EMessageType resolvedName{NMail::MT_UNDEFINED};
            TryFromString(name, resolvedName);
            if (resolvedName != NMail::MT_UNDEFINED) {
                types.push_back(resolvedName);
            }
        }
    }

    for (const auto& [deliveryId, rcpt]: recipients) {
        bool found = false;
        for (const auto& label: rcpt.params.labels) {
            if (label.type == "social") {
                domain_label = label.name;
                message.so_domain_label = label.name;
                found = true;
                break;
            }
        }
        if (found) {
            break;
        }
    }
}

}
