#pragma once

#include <mail/notsolitesrv/src/context.h>
#include <mail/notsolitesrv/src/envelope.h>
#include <mail/notsolitesrv/src/message/message.h>
#include <mail/notsolitesrv/src/types/common.h>
#include <mail/notsolitesrv/src/types/email_address.h>
#include <mail/notsolitesrv/src/user/storage.h>

#include <mimeparser/rfc2822address.h>

#include <util/generic/strbuf.h>

#include <boost/optional.hpp>

#include <cstdint>
#include <map>
#include <optional>
#include <string>
#include <utility>
#include <vector>

namespace NNotSoLiteSrv::NMetaSaveOp {

struct TFolderPath {
    static constexpr char DEFAULT_DELIM = '/';
    std::string path;
    std::string delim{DEFAULT_DELIM};
};

struct TFolder {
    boost::optional<std::string> fid;
    boost::optional<TFolderPath> path;
};

struct TLabel {
    TLabel(const std::string& name, const std::string& type): name(name), type(type) {}
    std::string name;
    std::string type;
};

struct TUser {
    std::optional<TOrgId> org_id;
    uint64_t uid;
    boost::optional<std::string> suid;
    std::string db;
    TEmailAddress email;
    std::string country;
    bool is_phone_confirmed = false;
    time_t born_date = 0;
    int32_t karma = 0;
    int32_t karma_status = 0;
    std::optional<bool> opt_in_subs_enabled;
};

enum class ENoSuchFolderAction {
    Fail = 1,
    FallbackToInbox,
    Create
};

using TEmailAddressList = std::vector<TEmailAddress>;
struct TParams {
    boost::optional<TEmailAddressList> bcc;
    boost::optional<TFolder> folder;
    std::vector<TLid> lids;
    std::vector<TLabel> labels;
    std::vector<std::string> label_symbols;
    boost::optional<std::string> tab;
    boost::optional<std::string> stid;
    boost::optional<size_t> offset_diff;
    boost::optional<std::string> old_mid;
    bool use_domain_rules = false;
    bool use_filters = true;
    bool ignore_duplicates = false;
    bool remove_duplicates = false;
    bool mailish = false;
    bool disable_push = false;
    bool store_as_deleted = false;
    bool spam = false;
    bool imap = false;
    ENoSuchFolderAction no_such_folder_action = ENoSuchFolderAction::FallbackToInbox;
    boost::optional<uint64_t> external_imap_id;
    bool process_as_subscription = false;
};

struct TEnvelope {
    std::string remote_ip;
    std::string remote_host;
    std::string helo;
    std::string mail_from;
    std::vector<std::string> recipients;
    time_t received_date = 0;
};

struct THidComparator {
    bool operator()(const THid& lhs, const THid& rhs) const;
};

struct TPart {
    std::string content_type;
    std::string content_subtype;
    std::string charset;
    std::string encoding;
    uint64_t offset = 0;
    uint64_t length = 0;
    boost::optional<std::string> boundary;
    boost::optional<std::string> name;
    boost::optional<std::string> content_disposition;
    boost::optional<std::string> file_name;
    boost::optional<std::string> content_id;
    boost::optional<std::string> data;
};

struct TAttachment {
    std::string name;
    std::string type;
    uint64_t size = 0;
};

using TPartMap = std::map<THid, TPart, THidComparator>;
using TReferences = std::vector<std::string>;
using TAttachmentMap = std::map<THid, TAttachment, THidComparator>;
struct TMessage {
    std::string subject;
    TEmailAddressList from;
    TEmailAddressList to;
    std::string message_id;
    uint64_t size = 0;
    TPartMap parts;
    boost::optional<TEmailAddressList> cc;
    boost::optional<std::string> sender;   // FIXME: TEmailAddress
    boost::optional<std::string> reply_to; // FIXME: TEmailAddress
    boost::optional<std::string> in_reply_to;
    boost::optional<TReferences> references;
    boost::optional<TAttachmentMap> attachments;
    time_t date = 0;
    boost::optional<std::string> so_domain_label;
    bool spam = false;
};

struct TRecipient {
    TUser user;
    TParams params;
};

using TRecipientMap = std::map<TDeliveryId, TRecipient>;
struct TRequest {
    TRequest() = default;
    TRequest(
        TContextPtr ctx,
        TMessagePtr message,
        const NNotSoLiteSrv::TEnvelope& envelope,
        NUser::TStorage& userStorage,
        bool updateOuterData = true);

    TMessage message;
    TEnvelope envelope;
    TRecipientMap recipients;
    std::string session_id;
    std::string stid;
    boost::optional<std::string> domain_label;
    std::vector<uint32_t> types;
    bool sync_dlv = false;
};

boost::optional<TEmailAddress> ToEmailAddress(const rfc2822::address_pair_t& addr);
boost::optional<TEmailAddress> ToEmailAddress(const std::string& straddr);
template<typename TList, typename TIter> void FillAddressList(const TList& from, TIter to);
uint64_t CalculateAttachmentSize(const std::string& encoding, const TStringBuf& body);
size_t GetPartOffset(off_t originalOffset, off_t offsetDiff);
void FindAndUpdateBestTextPart(const TMessagePtr message, TPartMap& parts);

void FindPartsAndAttachments(
    TMessagePtr msg,
    TPartMap& parts,
    TAttachmentMap& attachments);

TEnvelope CreateEnvelope(
    TMessagePtr message,
    const NNotSoLiteSrv::TEnvelope& envelope,
    const NUser::TStorage& userStorage);

TMessage CreateMessage(TContextPtr ctx, TMessagePtr origMessage);

std::pair<TFolder, ENoSuchFolderAction> CreateFolder(
    const TXYandexHint& hint,
    bool isSpam,
    bool isMailish);

void UpdateLabelSymbolsFromMixed(std::vector<std::string>& labelSymbols, int64_t mixed);

void UpdateLabelsInParams(
    TContextPtr ctx,
    TParams& params,
    const TXYandexHint& hint,
    bool isSpam,
    bool hasAttachments,
    bool sharedStid);

TUser CreateUser(const std::string& email, const NUser::TUser& recipient);

TParams CreateParams(
    TContextPtr ctx,
    const NUser::TUser& user,
    const TMessagePtr& message,
    bool isMailish,
    bool hasAttachments,
    bool sharedStid);

TRecipientMap CreateRecipients(TContextPtr ctx, NUser::TStorage& userStorage, TMessagePtr message,
    bool hasAttachments, bool updateOuterData);

bool HasAttachments(const TMessage& message);

}

namespace std {

inline string to_string(NNotSoLiteSrv::NMetaSaveOp::ENoSuchFolderAction val) {
    switch (val) {
        case NNotSoLiteSrv::NMetaSaveOp::ENoSuchFolderAction::Fail:
            return "fail";
        case NNotSoLiteSrv::NMetaSaveOp::ENoSuchFolderAction::FallbackToInbox:
            return "fallback_to_inbox";
        case NNotSoLiteSrv::NMetaSaveOp::ENoSuchFolderAction::Create:
            return "create";
    }
}

}
