#include "response.h"

#include <mail/notsolitesrv/src/tskv/logger.h>

namespace NNotSoLiteSrv::NMetaSaveOp {

namespace {

std::string MakeMissingFieldMsg(const std::string& deliveryId, const std::string& errorField) noexcept {
    std::stringstream msg;
    msg
        << "mdbsave response dlv_id=" << deliveryId
        << " does not contain '" << errorField << "'";
    return msg.str();
}

void SetErrorResult(NUser::TStorage& userStorage) {
    for (auto& [login, user]: userStorage.GetFilteredUsers(NUser::Found)) {
        if (!user.DeliveryResult.ErrorCode || !NError::IsPermError(user.DeliveryResult.ErrorCode)) {
            user.DeliveryResult.ErrorCode = EError::MetaSaveOpTemporaryError;
        }
    }
}

}

void UpdateRecipientsStatusFromResponse(TContextPtr ctx, NUser::TStorage& userStorage,
    const TResponse& response) noexcept
{
    try {
        for (const auto& [deliveryId, mdbResponse]: response.mdb_responses) {
            for (auto& [login, user]: userStorage.GetFilteredUsers(NUser::Found) | NUser::ByDeliveryId(deliveryId)) {
                const auto& mdbCommit = mdbResponse.mdb_commit;
                if (mdbCommit == boost::none) {
                    NSLS_LOG_ERROR(ctx, logdog::message="mdb_commit node not found");
                    if (mdbResponse.status == EStatus::PermanentError) {
                        user.DeliveryResult.ErrorCode = EError::MetaSaveOpPermanentError;
                    } else {
                        user.DeliveryResult.ErrorCode = EError::MetaSaveOpTemporaryError;
                    }
                    continue;
                }

                if (mdbCommit->uid != user.Uid) {
                    throw std::runtime_error("Invalid delivery id <-> uid");
                }

                switch (mdbCommit->status) {
                    case EStatus::Ok:
                        break;
                    case EStatus::TemporaryError:
                        [[ fallthrough ]];
                    case EStatus::Unknown:
                        user.DeliveryResult.ErrorCode = EError::MetaSaveOpTemporaryError;
                        break;
                    case EStatus::PermanentError:
                        user.DeliveryResult.ErrorCode = EError::MetaSaveOpPermanentError;
                        break;
                }

                const auto& uid = user.Uid;
                const auto& dlvId = deliveryId;
                if (user.DeliveryResult.ErrorCode) {
                    std::stringstream msg;
                    msg
                        << "mdbsave error for uid=" << uid << ", email=" << login
                        << ": " << (mdbCommit->description ? *mdbCommit->description : "unknown error");
                    NSLS_LOG_ERROR(ctx, logdog::message=msg.str());
                } else if (!mdbCommit->mid) {
                    NSLS_LOG_ERROR(ctx, log::uid=uid, logdog::message=MakeMissingFieldMsg(dlvId, "mid"));
                } else if (!mdbCommit->tid) {
                    NSLS_LOG_ERROR(ctx, log::uid=uid, logdog::message=MakeMissingFieldMsg(dlvId, "tid"));
                } else if (!mdbCommit->folder) {
                    NSLS_LOG_ERROR(ctx, log::uid=uid, logdog::message=MakeMissingFieldMsg(dlvId, "folder"));
                } else if (!mdbCommit->labels) {
                    NSLS_LOG_ERROR(ctx, log::uid=uid, logdog::message=MakeMissingFieldMsg(dlvId, "labels"));
                } else {
                    user.DeliveryResult.Mid = *mdbCommit->mid;
                    user.DeliveryResult.ImapId = std::to_string(*mdbCommit->imap_id);
                    user.DeliveryResult.IsDuplicate = *mdbCommit->duplicate;
                    user.DeliveryResult.Tid = *mdbCommit->tid;
                    user.DeliveryResult.Fid = mdbCommit->folder->fid;
                    if (mdbCommit->folder->name) {
                        user.DeliveryResult.FolderName = *mdbCommit->folder->name;
                    }
                    if (mdbCommit->folder->type_code) {
                        user.DeliveryResult.FolderTypeCode = *mdbCommit->folder->type_code;
                    }
                    if (mdbCommit->folder->type) {
                        user.DeliveryResult.FolderType = *mdbCommit->folder->type;
                    }
                    for (const auto& label: *mdbCommit->labels) {
                        user.DeliveryResult.Lids.emplace_back(label.lid);
                        if (label.symbol) {
                            user.DeliveryResult.LabelSymbols.emplace_back(*label.symbol);
                        }
                    }

                    if (mdbResponse.rule_ids) {
                        user.DeliveryResult.FilterIds = *mdbResponse.rule_ids;
                    }

                    if (mdbResponse.forward) {
                        std::transform(
                            mdbResponse.forward->begin(),
                            mdbResponse.forward->end(),
                            std::back_inserter(user.DeliveryResult.Forwards),
                            [](const auto& fwd) { return fwd.address; });
                    }

                    if (mdbResponse.notify) {
                        std::transform(
                            mdbResponse.notify->begin(),
                            mdbResponse.notify->end(),
                            std::back_inserter(user.DeliveryResult.Notifies),
                            [](const auto& notify) { return notify.address; });
                    }

                    if (mdbResponse.reply) {
                        std::transform(
                            mdbResponse.reply->begin(),
                            mdbResponse.reply->end(),
                            std::back_inserter(user.DeliveryResult.AutoReplies),
                            [](const auto& reply) { return std::make_pair(reply.address, reply.body); });
                    }
                }

                break;
            }
        }
    } catch (const std::exception& ex) {
        NSLS_LOG_ERROR(ctx, logdog::message="Error while parsing MetaSaveOp response", logdog::exception=ex);
        SetErrorResult(userStorage);
    }
}

}
